/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: tfile.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * tfile.c -- file/string reader/tokeniser
 * Robin Kirkham, June 1996
 */
#include "internal.h"
#include <ctype.h>

    /* size of argument lists */
#define MAXLINE	1024
#define MAXARGS	100

#undef TEST

    /* type of function that gets a character */
typedef char (* IIO_CHFN)(void *);

    /* scanner mode */
typedef enum {
    iio_tmode_comment,
    iio_tmode_space,
    iio_tmode_token,
    iio_tmode_stoken
} IIO_TSTATE;


HIDDEN char iio_tfile_mgetc(void *pstring) {
    /*
     * Get the next character from a string, pointed to by a pointer
     * pointed to by pstring
     */

    /* yes, it works */
    return (*(*((char**)pstring))++);
}


HIDDEN char iio_tfile_fgetc(void *pfile) {
    /*
     * Get the next character from a string, pointed to by a pointer
     * pointed to by pstring
     */
    return iio_file_getc(*(IIO_FILE *)pfile);
}

#ifdef TEST
HIDDEN IIO_STATUS iio_tfile_test(char *argv[]) {
    /*
     * For testing the scanner. Prints its arguments
     */
    int count;

    for (count = 0; argv[count]; ++count)
	printf("<%s> ", argv[count]);
    printf("\n");
    return iio_status_ok;
}
#endif


    /* macros used in iio_tfile_scan() */
#define ENDTOKEN { \
	*point++ = '\0'; \
	argv[++argc] = point; \
	++linelen; \
    }

#define ADDTOKEN(ch) { \
	if (argc == 0) argc = 1; \
	*point++ = ch; \
	++linelen; \
    }

#ifdef TEST
#define EXECLINE { \
	argv[argc] = NULL; \
	if (argc > 1) \
	    iio_tfile_test(argv); \
	    argc = 0; \
    }
#else
#define EXECLINE { \
	argv[argc] = NULL; \
	if (argc > 1) \
	    iio_eret( texec(argv) ); \
	    argc = 0; \
    }
#endif


HIDDEN IIO_STATUS iio_tfile_scan(
    char *name, void *source, IIO_CHFN getc, IIO_TEXEC texec
) {
    /* 
     * Scan and tokenize a file, assemble the space-separated tokens into
     * logical line argument lists, then call texec on each line. The scanner
     * is a state machine, dealing with the file character by character.
     * The scanner permits logical lines to continue over physical lines, 
     * if the last non-space character in a line is \; comments beginning with
     * a #; and strings delimited by ".
     */
    char *argv[MAXARGS];
    char ch, buffer[MAXLINE], *point = buffer;
    int argc = 0, linelen = 0, lineno = 1;
    IIO_TSTATE mode = iio_tmode_space;
    int slash = 0;


    /* get characters until end-of-file (character 0) */
    while ((ch = getc(source))) {

	/* if no argv[0] (reference line), make one up */
	if (argc == 0) {
	    iio_slog(buffer, "%.30s, line %d", name, lineno);
	    linelen = iio_string_len(buffer) + 1;
	    point = buffer + linelen;
	    *point = '\0';

	    argv[0] = buffer;
	    argv[1] = point;
	    argv[2] = NULL;
	}

	/* check line line length */
	if (argc >= MAXARGS - 1)
	    return iio_fatal("Too many tokens");
	if (linelen >= MAXLINE - 2)
	    return iio_fatal("Line too long");

	/* decrement the slash flag */
	if (slash)
	    --slash;

	/* process character according to mode */
	switch (mode) {
	case iio_tmode_space:
	    /*
	     * In space mode, eat whitespace until a #, \n, \, or printable,
	     * then change mode to the most appropriate new mode 
	     */
	    switch (ch) {
	    case '#':
		/* exec current line, then enter comment mode */
		if (slash == 0)
		    EXECLINE;
		mode = iio_tmode_comment;
		break;

	    case ' ':
	    case '\t':
		/* spaces preserve slash-status */
		++slash;
		break;

	    case '\n':
		/* end of line: execute it, unless slash-escaped */
		if (slash == 0)
		    EXECLINE;
		++lineno;
		break;

	    case '"':
		/* start of string token */
		mode = iio_tmode_stoken;
		break;

	    case '\\':
		/* start of backslash-escape */
		slash = 2;
		break;

	    default:
		/* any other printable--add a (new) token */
		if (isgraph((int)ch)) {
		    ADDTOKEN(ch);
		    mode = iio_tmode_token;
		}
		break;
	    }
	    break;


	case iio_tmode_comment:
	    /*
	     * In comment mode, eat everything until a \n
	     */
	    switch (ch) {
	    case '\n':
		/* end of line */
		mode = iio_tmode_space;
		++lineno;
		break;

	    default:
		/* gulp */
		break;
	    }
	    break;


	case iio_tmode_token:
	    /*
	     * In token mode, addend chars to current token, end the token
	     * at a space, execute the line at a newline.
	     */
	    switch (ch) {
	    case ' ':
	    case '\t':
		/* end of token */
		ENDTOKEN;
		mode = iio_tmode_space;
		break;

	    case '\n':
		/* end of line: execute it */
		ENDTOKEN;
		EXECLINE;
		mode = iio_tmode_space;
		++lineno;
		break;

	    case '"':
		/* end current token, start new string token */
		ENDTOKEN;
		mode = iio_tmode_stoken;
		break;

	    case '\\':
		/* start of backslash-escape */
		slash = 2;
		break;

	    default:
		/* any other printable--add a (new) token */
		if (isgraph((int)ch))
		    ADDTOKEN(ch);
		break;
	    } 
	    break;


	case iio_tmode_stoken:
	    /*
	     * In string token mode, addend chars to current token (including
	     * spaces and tabs), end the token at a " or \n, execute the
	     * line at a \n.
	     */
	    switch (ch) {
	    case '\n':
		/* end of line: execute it */
		ENDTOKEN;
		EXECLINE;
		mode = iio_tmode_space;
		++lineno;
		break;

	    case '"':
		/* end current string token */
		ENDTOKEN;
		mode = iio_tmode_space;
		break;

	    default:
		/* any other printable or space--add a (new) token */
		if (isgraph((int)ch) || ch == ' ' || ch == '\t')
		    ADDTOKEN(ch);
		break;
	    }
	    break;
	}
    } 

    return iio_status_ok;
}


IIO_STATUS iio_tfile(IIO_TEXEC texec, IIO_TFILE method, ...) {
    /*
     * Parse the token-file or token-string using the given method and 
     * texec() function. The arguments are:
     *
     *	    iio_tfile_file, IIO_FILE file, char *name
     *	    iio_tfile_filename, char *filename
     *	    iio_tfile_string, char *string, char *name
     */
    va_list ap; 
    va_start(ap, method);
    iio_eret( iio_tfile_stdarg(texec, method, ap) );
    return iio_status_ok;
}

IIO_STATUS iio_tfile_stdarg(IIO_TEXEC texec, IIO_TFILE method, va_list ap) {
    /*
     * Parse the token-file or token-string using the given method and 
     * texec() function, getting the data from the arg pointer
     */
    char *name, *string;
    IIO_FILE file;

    switch (method) { 
    case iio_tfile_file:
	file = va_arg(ap, IIO_FILE);
	name = va_arg(ap, char*);
	iio_eret(
	    iio_tfile_scan(
		name,
		(void *)&file,
		iio_tfile_fgetc,
		texec
	    )
	);
	break;

    case iio_tfile_filename: 
	name = va_arg(ap, char*);
	iio_eret( iio_file_open(name, iio_fattr_rdonly, &file) );
	iio_eret(
	    iio_tfile_scan(
		name,
		(void *)&file,
		iio_tfile_fgetc,
		texec
	    )
	);
	iio_eret( iio_file_close(file) );
	break;

    case iio_tfile_string:
	string = va_arg(ap, char*);
	name = va_arg(ap, char*);
	iio_eret(
	    iio_tfile_scan(
		name,
		(void *)&string,
		(IIO_CHFN)iio_tfile_mgetc,
		texec
	    )
	);
    }
    return iio_status_ok;
}
