/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: resolve.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * resolve.c -- address resolver
 * Robin Kirkham, July 1996
 */
#include "internal.h"


IIO_STATUS iio_resolve_physical(
    IIO_OPEN *chan, IIO_OP space, IIO_SIZE size, unsigned laddr, void **paddr
) {
    /*
     * Resolve the given local address using operation op on the given channel
     * to obtain a PHYSICAL address. Return the physical address into paddr
     */
    void *paddr1 = (void *)laddr;

    /* check parameters */ 
    if (!chan || chan->magic != iio_magic_open)
	return iio_fatal("NULL or bad channel pointer");
    if (!paddr)
	return iio_fatal("NULL physical address return pointer");

    /*
     * Use the channel to resolve address to physical. Note that the operation
     * code is the arithmatic OR of the space enum and the space operation code.
     * The operation overwrites the input data
     */
    iio_eret( iio_operate_addr(chan, space | size, &paddr1) );
    *paddr = paddr1;

    return iio_status_ok;
}


IIO_STATUS iio_resolve(
    IIO_OPEN *chan, IIO_OP space, IIO_SIZE size, unsigned laddr, void **vaddr
) {
    /*
     * Resolve the given local address using operation op on the given channel
     * to obtain a physical address, then use the physical address lookup 
     * function iio_vmap_ptov() to get the virtual address for it.
     *
     * Note that the size of the segment might not be the same, because the
     * channel resolver might expand the space (insert gaps, like the NuBus
     * IP carriers do). For this reason we separately resolve the bottom and
     * top addresses, and compute the size as the difference in the resolved
     * addresses
     */
    IIO_MAPTYPE type;
    void *paddr1 = (void *)laddr;
    void *paddr2 = (void *)(laddr + (unsigned)size/(unsigned)iio_size_8);

    /* check parameters */ 
    if (!chan || chan->magic != iio_magic_open)
	return iio_fatal("NULL or bad channel pointer");

    /*
     * Use the channel to resolve address to physical. Note that the operation
     * code is the arithmatic OR of the space enum and the space operation code.
     * The operation overwrites the input data
     */
    iio_eret( iio_operate_addr(chan, space | size, &paddr1) );
    iio_eret( iio_operate_addr(chan, space | size, &paddr2) );

    /* resolve physical to virtual */
    iio_eret( iio_map_type(space, &type) );
    iio_eret( iio_map_ptov(type, paddr1, (unsigned)(paddr2 - paddr1), vaddr) );

    return iio_status_ok;
}


IIO_STATUS iio_resolve_list(IIO_OPEN *chan, IIO_OP space, ...) {
    /*
     * Convenience form of iio_resolve(), where a large number of registers
     * have to be resolved againas the same address space. Following the
     * space argument is a list of size, laddr, **vaddr triplets, which are
     * passed to iio_resolve(). The list is terminated by a zero size
     */
    va_list ap;
    va_start(ap, space);

    while (1) {
	IIO_SIZE size = va_arg(ap, IIO_SIZE);
	unsigned laddr;
	void **vaddr;

	/* check the size argument for sanity */
	switch (size) {
	case 0:
	    /* end of list */
	    return iio_status_ok;

	case iio_size_8:
	case iio_size_16:
	case iio_size_32:
	case iio_size_64:

	    /* resolve the triplet */
	    laddr = va_arg(ap, unsigned);
	    vaddr = va_arg(ap, void **);
	    iio_eret( iio_resolve(chan, space, size, laddr, vaddr) );
	    break;

	default:
	    /* something is wrong */
	    return iio_fatal("Bogus IIO_SIZE argument");
	}
    }
    return iio_status_ok;
}
