/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: phantom.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * phantom.c -- the phantom device driver 
 */
#include "internal.h"

    /* the state structure */
struct IIO_MSTATE {
    /*
     * MSTATE is an array of ints, one for each channel, which stores
     * the last write to the channel, so read and readback can be
     * (sort of) simulated. The zero-length array might be a GCC extension
     */
    int data[0];
};

struct IIO_MREG {
    /*
     * Normally the register structure would hold configuration details 
     * from the options, plus pointers to the device registers. Here there
     * is no hardware, so just configuration is stored
     */
    IIO_CHTYPE type;
    unsigned int width;
    uint16_t number;
};


HIDDEN IIO_STATUS iio_phantom_operate(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * The phantom driver doesn't do anything, but it must at least store the
     * "output" state, so the output read-back is simulated, at least vaguely.
     * Input also just reads back the output state. The state is stored in
     * the state structure, in an array, after the initial elements.
     */
    unsigned seqno;

    for (seqno = first ; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    state->data[seqno] = iio_data_get(opnode, seqno);
	    break;

	case iio_op_clear:
	    state->data[seqno] = 0;
	    break;

	case iio_op_read:
	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, state->data[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}

    return iio_status_ok;
}


IIO_STATUS iio_phantom_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialisation is simply clearing the a state structure. Unusually,
     * the phantom driver has a variable sized state block, as the number of
     * channels is defined by an option, not fixed in the module hardware.
     * The number of channels is in the reg block
     */
    int count;

    /* zero the state structure */
    for (count = 0; count < reg->number; ++count)
	state->data[count] = 0;

    return iio_status_ok;
}


IIO_STATUS iio_phantom_install(IIO_MODULE *module, char *argv[]) {
    /*
     * The phantom driver simulates a module of channels of a certain type.
     * The number, type and width of the channels are specified using
     * options -number, -type and -width. The -phantom flag is read and
     * ignored
     */
    char *type = "do";
    IIO_MREG *reg;
    IIO_STATUS stat;

    /* obtain a register structure and store the details */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    reg->number = 1;
    reg->width = channelwidth(int);

    /* get the arguments */
    iio_arg(argv, "number", iio_arg_uint16, &reg->number);
    iio_arg(argv, "type", iio_arg_string, &type);

    /* allocate a state structure for the readbacks */
    iio_eret( iio_module_state(module, sizeof(uint32_t)*reg->number) );

    /* look up channel type/width argument */
    iio_fret( stat = iio_chtype_find(type, &reg->type, &reg->width) );
    if (stat) {
	iio_log(
	    "%s: Argument to `-type`: %s\n",
	    argv[0],
	    iio_emessage_get()
	);
	return stat;
    }

    /* register the channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    reg->type,
	    reg->width,
	    reg->number,
	    iio_phantom_operate,
	    NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_phantom(void) {
    /*
     * Register the phantom driver with IIO, by calling iio_minfo().
     * Called once, usually from the install list. Note a module can
     * install itself under a number of names
     */
    return iio_minfo(
	"phantom",			/* Our module ID */
	"<phantom>",			/* Our name */
	"$Revision: 2222 $",			/* File RCS ID */
	iio_multi_yes,			/* we can be multiply installed */
	iio_phantom_install,		/* the install function */
	iio_phantom_init		/* the init function */
    );
}
