/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: tty.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * tty.c -- generic tty functions
 * Robin Kirkham, January 1997
 *
 * This sub-module is intended to be textually included into 
 * one of the os-specific modules 
 *
 * This is the unix/termios version. It supports the subset of the termios
 * magic that it likely to be needed by serial IO devices, and which can
 * be implemented using all the operating systems
 */


HIDDEN IIO_STATUS iio_tty_basic(
    struct termios *termios,
    unsigned baud, unsigned nchar, unsigned nstop,
    IIO_TTYPAR parity, IIO_TTYFLOW flow, IIO_TTYMAP map
) {
    /*
     * Set up TTY basic modes mode with the given parameters.
     */
    speed_t baudcode;

    /* set the input modes */
    termios->c_iflag = 
	((parity & iio_ttypar_odd) ? INPCK : 0) |
	((parity & iio_ttypar_even) ? INPCK : 0) |
	((parity & iio_ttypar_ignore) ? IGNPAR : 0) |
	((parity & iio_ttypar_mark) ? PARMRK : 0) |
	((flow & iio_ttyflow_xonxoff) ? (IXON | IXOFF) : 0) |
	((flow & iio_ttyflow_xonany) ? (IXON | IXOFF | IXANY) : 0) |
	((map & iio_ttymap_instrip) ? ISTRIP : 0) |
	((map & iio_ttymap_incrnl) ? ICRNL : 0);

    /* set the output modes */
    termios->c_oflag = 0;


    /* set the control modes */
    termios->c_cflag = CREAD | CLOCAL;

    /* character length */
    switch (nchar) {
    case 5: termios->c_cflag |= CS5; break;
    case 6: termios->c_cflag |= CS6; break;
    case 7: termios->c_cflag |= CS7; break;
    case 8: termios->c_cflag |= CS8; break;
    default:
	return iio_error("Bad character length");
    }

    /* stop bits */
    switch (nstop) {
    case 1: termios->c_cflag |= 0; break;
    case 2: termios->c_cflag |= CSTOPB; break;
    default:
	return iio_error("Bad stop bits length");
    }

    /* baud rate */
    switch (baud) {
    case 50: baudcode = B50; break;
    case 75: baudcode = B75; break;
    case 110: baudcode = B110; break;
    case 134: baudcode = B134; break;
    case 150: baudcode = B150; break;
    case 200: baudcode = B200; break;
    case 300: baudcode = B300; break;
    case 600: baudcode = B600; break;
    case 1200: baudcode = B1200; break;
    case 1800: baudcode = B1800; break;
    case 2400: baudcode = B2400; break;
    case 4800: baudcode = B4800; break;
    case 9600: baudcode = B9600; break;
    case 19200:	baudcode = B19200; break;
    case 38400:	baudcode = B38400; break;
    default:
	return iio_error("Bad baud rate");
    }
    if (cfsetospeed(termios, baudcode) < 0)
	return iio_error(NULL);

    termios->c_cflag |= 
	((flow & iio_ttyflow_hupcl) ? HUPCL : 0) |
#ifdef CRTSCTS
	((flow & iio_ttyflow_rtscts) ? CRTSCTS : 0) |
#endif
	((parity & iio_ttypar_odd) ? (PARENB | PARODD) : 0) |
	((parity & iio_ttypar_even) ? PARENB : 0);

    /* set the line modes */
    termios->c_lflag = 0;

    return iio_status_ok;
}


IIO_STATUS iio_tty_line(
    IIO_FILE tty, 
    unsigned baud, unsigned nchar, unsigned nstop,
    IIO_TTYPAR parity, IIO_TTYFLOW flow, IIO_TTYMAP map,
    IIO_TTYECHO echo
) {
    /*
     * Set up TTY to be in line mode (canonical input processing)
     */
    struct termios termios;

    /* read the current TTY driver settings, largely ignored */
    if (tcgetattr(tty, &termios) < 0)
	return iio_error(NULL);

    /* set most characteristics */
    iio_eret( iio_tty_basic(&termios, baud, nchar, nstop, parity, flow, map) );

    /* set canonical line modes */
    termios.c_lflag =
	ISIG | ICANON |
	((echo & iio_ttyecho_echo) ? ECHO : 0);

    /* write settings to tty driver */
    if (tcsetattr(tty, TCSANOW, &termios) < 0)
	return iio_error(NULL);
    return iio_status_ok;
}


IIO_STATUS iio_tty_raw(
    IIO_FILE tty, 
    unsigned baud, unsigned nchar, unsigned nstop,
    IIO_TTYPAR parity, IIO_TTYFLOW flow, IIO_TTYMAP map,
    unsigned char min, unsigned char time
) {
    /*
     * Set up TTY for raw (non-canonical) local mode with
     * the given parameters.
     */
    struct termios termios;

    /* read the current TTY driver settings, largely ignored */
    if (tcgetattr(tty, &termios) < 0)
	return iio_error(NULL);

    /* set most characteristics */
    iio_eret( iio_tty_basic(&termios, baud, nchar, nstop, parity, flow, map) );

    /* set raw mode min/time parameters */
    termios.c_cc[VMIN] = min;
    termios.c_cc[VTIME] = time;

    /* write settings to tty driver */
    if (tcsetattr(tty, TCSANOW, &termios) < 0)
	return iio_error(NULL);
    return iio_status_ok;
}


IIO_STATUS iio_tty_send(IIO_FILE tty, char *buffer, unsigned len) {
    /*
     * Send a message of len caracters to the given tty file
     */
    if (write(tty, buffer, len) != len)
	return iio_error(NULL);
    return iio_status_ok;
}

IIO_STATUS iio_tty_recv(
    IIO_FILE tty, char *buffer, unsigned maxlen, unsigned *len, unsigned ms
) {
    /*
     * Read a line (message) of up to maxlen character from the given tty,
     * until either an end-of-line is received (in cooked mode) or ms 
     * milliseconds has elapsed
     */
    int rlen;

    if ((rlen = read(tty, buffer, maxlen)) <= 0)
	return iio_error(NULL);
    if (len)
	*len = rlen;

    return iio_status_ok;
}

