/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: linux.c 3094 2008-05-16 06:05:18Z roy029 $
 *
 * linux.c -- OS module for Linux
 * Robin Kirkham, July 1996
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>
#include <termios.h>
#include <math.h>
#include <sys/mman.h>

#include <sys/io.h>

#include "config.h"

#ifndef SHARED_MUTEXES
#include <semaphore.h>
#endif

#include "../iio.h"
#include "../internal.h"


#define IIO_LYNXOS

    /* include the libc wrappers */
#include "mem.c"
#include "decode.c"
#include "string.c"
#include "slog.c"
#include "file.c"
#include "round.c"

    /* include the generic sub-modules */
#include "roughdelay.c"
#include "emessage.c"
#include "log.c"
#include "exec.c"
#include "tty.c"

HIDDEN unsigned iio_pagesize = 0;
HIDDEN IIO_BOOL iio_first = iio_bool_true;
HIDDEN IIO_SENTINEL *iio_sentinel = NULL;

/*
 * ------------------------------------------------ OS-SPECIFIC INITIALISATION
 */

IIO_STATUS iio_osinit(IIO_INFOFN list[], IIO_IFLAG flags) {
    /*
     * This is called by iio_init().
     *
     * IIO initialisation is complicated on LynxOS. The shared-memory segments
     * are used to store device state and for mutex locks, except for the 
     * initialisation mutex, which is an advisory file lock on the
     * configuration file.
     *
     * When we are called, we first open and obtain an exclusive lock on the
     * file /etc/iio.conf. We block if it is not available. Once obtained, we
     * test for the existance of the shared-memory segments. This tells us if
     * we are the initial IIO run or not.
     *
     * ...
     */
    IIO_FILE file = -1;
    struct stat stat;
    int shmemfd = -1;

    /* an important number */
    /*
    iio_pagesize = getpagesize();
    */
    iio_pagesize = 4096;
    
    /* open the configuration file */
    iio_eret( iio_file_open("/etc/iio.conf", iio_fattr_rdonly, &file) );

    /* obtain an exclusive lock on it */
    if (flock(file, LOCK_EX) != 0) 
	return iio_error(NULL);

    /* stat the file to get its mtime */
    if (fstat(file, &stat) != 0) 
	return iio_error(NULL);


    /* 
     * Attempt to *remove* the first block of shared memory, which we know
     * will have the name "iio-shmem-0". If the call fails with errno == EBUSY
     * we know there is another IIO process running, and hence we are not the
     * first (the remove will not cause harm).
     *
     * Otherwise, either there is no shared memory of that name, or there
     * was an unattached block of that name. In either case, we should assume
     * we are the first IIO process.
     */
    /*
    if ((smem_remove("iio-shmem-0") != 0) && (errno == EBUSY))
	iio_first = iio_bool_false;
    */
    /* For POSIX shared memory, this just wont work. For now, we just open
     * the block to see if it exists.
     */
    if ((shmemfd = shm_open("/iio-shmem-0", O_RDWR, 0666)) != -1) {
	iio_first = iio_bool_false;
	close (shmemfd);
    }


    /*
     * Establish the "sentinel" structure, which contains the mtime of the
     * config file, and the "mtime" of the IIO library itself. If this is
     * the initial process, initialise the sentinel. Otherwise, check the 
     * existing times against our ones, to make sure the shared memory
     * belongs to the the same version of the IIO library and config file
     */
    iio_eret( iio_shmem_alloc(sizeof(IIO_SENTINEL), (void **)&iio_sentinel) ); 

    if (iio_first) {

	/* initialise a new sentinel */
	iio_sentinel->magic = iio_magic_sentinel;
	iio_sentinel->config_time = (unsigned long)stat.st_mtime;
	iio_sentinel->library_time = iio_timestamp;

    } else {

	/* check the existing sentinel */
	if (iio_sentinel->magic != iio_magic_sentinel)
	    return iio_error("Shared memory has wrong magic number");
	if (iio_sentinel->config_time != (unsigned long)stat.st_mtime)
	    return iio_error("Shared memory is inconsistent with config file");
	if (iio_sentinel->library_time != iio_timestamp)
	    return iio_error("Shared memory is inconsistent with IIO version");
    }


    /* initialise the IIO library, install ind initialise drivers */
    iio_eret(
	iio_init_iio(
	    list,
	    flags,
	    iio_first,
	    iio_tfile_file,
	    file,
	    "/etc/iio.conf"
	)
    );

    /* release the lock */
    if (flock(file, LOCK_UN) != 0) 
		return iio_error(NULL);

    /* and close the file */
    iio_eret( iio_file_close(file) );

    return iio_status_ok;
}


/*
 * -------------------------------------------------------- PER-PROCESS MUTEXES
 */

IIO_STATUS iio_mutex_create(IIO_MUTEX **mutex) {
    /*
     * Create a process-mutex, and return the pointer to it.
     * These use the POSIX mutexes, which unfortunately don't have
     * priority inheritance. The memory for the mutex is obtained
     * from ordinary malloc()
     */
    pthread_mutex_t *new;
    pthread_mutexattr_t attr;

    if (! mutex)
        return iio_fatal("NULL mutex pointer");

    /* get memory for mutex object */ 
    iio_eret( iio_mem_alloc(sizeof(pthread_mutex_t), (void **)&new) );

    /* create mutex attribute thing */
    if (pthread_mutexattr_init(&attr) < 0)
        return iio_fatal(NULL);

    /* initialise the mutex */
    if (pthread_mutex_init(new, &attr) < 0)
	return iio_fatal(NULL);

    /* return pointer to new mutex */
    *mutex = (IIO_MUTEX *)new;

    return iio_status_ok;
}

IIO_STATUS iio_mutex_grab(IIO_MUTEX *mutex) {
    /*
     * Enter the mutex, potentially blocking
     */
    if (pthread_mutex_lock((pthread_mutex_t *)mutex) < 0)
	return iio_fatal(NULL);
    return iio_status_ok;
}

IIO_STATUS iio_mutex_drop(IIO_MUTEX *mutex) {
    /*
     * Exit the mutex
     */
    if (pthread_mutex_unlock((pthread_mutex_t *)mutex) < 0)
	return iio_fatal(NULL);
    return iio_status_ok;
}

IIO_STATUS iio_mutex_free(IIO_MUTEX *mutex) {
    /*
     * Delete the mutex, and release the memory used by it
     */
    if (pthread_mutex_destroy((pthread_mutex_t *)mutex) < 0)
	return iio_fatal(NULL);
    iio_eret( iio_mem_free((void *)mutex) );
    return iio_status_ok;
}


/*
 * ----------------------------------------------- SHARED (SYSTEM-WIDE) MUTEXES
 */

IIO_STATUS iio_shmutex_create(IIO_SHMUTEX **mutex) {
    /*
     * Create a globally shared mutex, and return the pointer to it.
     * The memory for the mutex is obtained from iio_shmem_alloc(), so
     * this mutex is shared among threads and processes. Hence, the mutex
     * itself only has to be created by the first IIO process
     *
     * It is otherwise the same as the other mutexes
     */

    /*
     * Here we use a mutex. Note that this will not work using a draft 4
     * pthread implementation!
     */
    pthread_mutex_t *new;
    pthread_mutexattr_t attr;

    if (! mutex)
        return iio_fatal("NULL mutex pointer");

    /* get memory for mutex object */ 
    iio_eret( iio_shmem_alloc(sizeof(pthread_mutex_t), (void **)&new) );

    /* if we are the first IIO process */
    if (iio_first) {

	/* create mutex attribute thing */
	if (pthread_mutexattr_init(&attr) < 0)
	    return iio_fatal(NULL);

	/* initialise the mutex */
	if (pthread_mutex_init(new, &attr) < 0)
	    return iio_fatal(NULL);
    }

    /* return pointer to new mutex */
    *mutex = (IIO_SHMUTEX *)new;

    return iio_status_ok;
}

IIO_STATUS iio_shmutex_grab(IIO_SHMUTEX *mutex) {

    /*
     * Grab a shared mutex. The shared mutexes are the same
     * as ordinary ones for this operation
     */
    return iio_mutex_grab(mutex);
}

IIO_STATUS iio_shmutex_drop(IIO_SHMUTEX *mutex) {

    /*
     * Drop a shared mutex. The shared mutexes are the same
     * as ordinary ones for this operation
     */
    return iio_mutex_drop(mutex);
}

/*
 * ----------------------------------------------- SHARED (SYSTEM-WIDE) MEMORY
 */

#define SHMEM_MAX 20
HIDDEN unsigned iio_shmem_serial = 0;
HIDDEN void *iio_shmem_next = NULL;
HIDDEN unsigned iio_shmem_size = 0;
HIDDEN void *iio_shmem_base[SHMEM_MAX];
HIDDEN void *iio_shmem_seg_size[SHMEM_MAX];

IIO_STATUS iio_shmem_alloc(unsigned int size, void **new) {
    /*
     * Allocate globally shared memory. This is obtained by calling the
     * LynxOS shared memory allocator smem_get(), which either creates a new
     * area of process-shared memory, or attaches to an existing one of the
     * same name and size.
     *
     * The whole shared state scheme of IIO on LynxOS depends on all processes
     * executing the same config file with the same library code, and so
     * exactly the same sequence of shared memory (and shared map) segments
     * will be requested. The first run creates them, and subsequent ones 
     * just attach to them. The code does not have to know which.
     *
     * This function allocates small areas from the pages from smem_get().
     * If the requested block won't fit in what remains, a new block is 
     * obtained. It is not a very efficient allocator, but it should work OK
     * as most IIO shared memory requests are small compared to pagesize.
     *
     * If this is the first IIO process, zero the new shared memeory
     */
    int shmemfd = -1;

    if (! new)
        return iio_fatal("NULL shared memory return pointer");
    if (! size)
        return iio_fatal("Request for zero sized shared memory");
   

    /* see if there is enough left in the previous block */
    if (size > iio_shmem_size) {
	char name[32];

	/* this limit will be abolished one day */
	if (iio_shmem_serial >= SHMEM_MAX)
	    return iio_fatal("Too many shared memory segments");

	/* round up size to next biggest page size */
	iio_shmem_size = (size/iio_pagesize + size%iio_pagesize ? 1 : 0)
	    * iio_pagesize;

	/* get a new block, or attach to old one */
	sprintf(name, "iio-shmem-%d", iio_shmem_serial);
	/*
	if (! (
	    iio_shmem_next = smem_get(
		name, iio_shmem_size, SM_READ | SM_WRITE
	    )
	))
	    return iio_fatal(NULL);
	*/ 
	if ((shmemfd = shm_open (name, O_RDWR, 0666)) == -1) {
	    if ((shmemfd = shm_open (name, O_RDWR | O_CREAT, 0666)) == -1)
		return iio_fatal(NULL);
	    if (ftruncate (shmemfd, iio_shmem_size) == -1)
		return iio_fatal(NULL);
	    if ((iio_shmem_next = mmap (0, iio_shmem_size,
					PROT_READ | PROT_WRITE,
					MAP_SHARED, shmemfd, 0)) == MAP_FAILED)
		return iio_fatal(NULL);
	    close (shmemfd);
	} else {
	    if ((iio_shmem_next = mmap (0, iio_shmem_size,
					PROT_READ | PROT_WRITE,
					MAP_SHARED, shmemfd, 0)) == MAP_FAILED)
		return iio_fatal(NULL);
	    close (shmemfd);
	}
					

	/* 
	 * we have to record the base address of the segment in an array, as
	 * the address has to be used to detach the segment, rather than the
	 * name. I don't understand the reason for this LynxOS wierdness. 
	 * It is a big nuisance
	 */
	iio_shmem_base[iio_shmem_serial++] = iio_shmem_next;
	iio_shmem_seg_size[iio_shmem_serial++] = (void *) iio_shmem_size;
    }

    /* carve a bit off the current/new block */
    *new = iio_shmem_next;

    /* clear it if required */
    if (iio_first) {
	char *point = iio_shmem_next;
	int count;

	for (count = 0; count < size; ++count)
		*point++ = 0;
    }

    /* reset next/remaining pointers */
    iio_shmem_next += size;
    iio_shmem_size -= size;

    return iio_status_ok;
}

IIO_STATUS iio_shmem_free(void *old) {
    /*
     * Free globally shared memory, at least from this process.
     * You can't actually free it, so we pretend
     */
    return iio_status_ok;
}

HIDDEN IIO_STATUS iio_shmem_done(void) {
    /*
     * Attempt to detach and remove all shared memory segments. This won't
     * affect any other processes using the segment
     */
    int count;

    for (count = 0; count < iio_shmem_serial; ++count) {
	char name[32];

	munmap (iio_shmem_base[count], (size_t) iio_shmem_seg_size[count]);
	sprintf(name, "iio-shmem-%d", count);
	shm_unlink (name);
    }
    return iio_status_ok;
}



/*
 * ---------------------------------- DIRECT MAPPED SHARED (SYSTEM-WIDE) MEMORY
 */

HIDDEN unsigned iio_shmap_serial = 0;

IIO_STATUS iio_shmap_alloc(
    void *paddr, unsigned int psize,
    void **pactual, void **vaddr, unsigned int *vsize
) {
    /*
     * Allocate a direct mapped shared memory segment. Ensure that the mapping
     * least enclosing the specified physical address and size exists, and
     * return into pactual the physical base address, into vaddr the
     * corresponding virtual address, and into vsize the actual size of the
     * mapping
     */
    unsigned size, rpsize;
    void *rpaddr, *rvaddr;
    char name[32];
    int fd;

    /* round out request to the LynxOS page size */
    rpaddr = (void *)(((unsigned)paddr/iio_pagesize) * iio_pagesize);
    rpsize = psize + (paddr - rpaddr);
    size = (rpsize/iio_pagesize + (rpsize%iio_pagesize ? 1 : 0)) * iio_pagesize;

    /* get a process-unique name for this mapping */
    sprintf(name, "iio-shmap-%d", iio_shmap_serial++);

    /* call LynxOS function to establish mapping */
    /*
    if (! (rvaddr = smem_create(
	name, rpaddr, size, SM_NCREAT | SM_READ | SM_WRITE
    )))
	iio_fatal(NULL);
    */
    if (( fd = open ("/dev/mem", O_RDWR)) == -1) {
        /*
        perror ("iio_shmap_alloc: open()");
	*/
	iio_fatal(NULL);
    }
    /*
    printf ("iio_shmap_alloc: paddr 0x%08X, psize %d, rpaddr 0x%08X, "
	    "size %d\n", (unsigned int) paddr, psize, (unsigned int) rpaddr,
	    size);
    */
    if ((rvaddr = mmap (NULL, (size_t) size, PROT_READ | PROT_WRITE,
			MAP_SHARED, fd, (off_t) rpaddr)) == MAP_FAILED) {
        /*
        perror ("iio_shmap_alloc: mmap()");
	*/
	iio_fatal(NULL);
    }
    close (fd);

    /* return actual physical base */
    if (pactual)
	*pactual = rpaddr;

    /* return virtual segment address and size */
    if (vaddr)
	*vaddr = rvaddr;
    if (vsize)
	*vsize = size;

    return iio_status_ok;
}

IIO_STATUS iio_shmap_free(void *vaddr, void *paddr, unsigned size) {
    /*
     * Free a direct mapped shared memory segment, specified by the virtual
     * and physical addresses and the size. LynxOS needs to know only the 
     * virtual address. We use smem_create() to detach the segment. Note that
     * this does not remove the segment itself (another process could still
     * be using it anyway). See iio_shmap_done() below
     */

    /* detach it (no error return) */
    /*
    smem_create(NULL, vaddr, 0, SM_DETACH);
    */
    munmap (vaddr, size);

    return iio_status_ok;
}

HIDDEN IIO_STATUS iio_shmap_done(void) {
    /*
     * Attempt to remove all shared map segments, assuming they have been
     * detached from this process at least. This might fail harmlessly
     */

    return iio_status_ok;
}


/*
 * --------------------------------------------------------------- PROBE MEMORY
 */

HIDDEN jmp_buf iio_probe_jmpbuf;

HIDDEN RETSIGTYPE iio_probe_handle(int sig) {
    /*
     * Signal handler for bus and segmentation errors generated
     * by iio_probe(). This sets the error message, and returns the error
     * code through a _longjmp() at the end. LynxOS does not seem to set
     * the code argument to anything useful
     */
    IIO_STATUS stat;

    switch (sig) {
    case SIGBUS:
        stat = iio_error("Bus error on probe");
	break;

    case SIGSEGV:
	stat = iio_error("Segmentation fault on probe");
	break;

    default:
	stat = iio_fatal("Unexpected signal on probe");
	break;
    }
    longjmp(iio_probe_jmpbuf, stat);
}

IIO_STATUS iio_probe(volatile void *vaddr, IIO_SIZE bytes, IIO_PTYPE ptype) {
    /*
     * Probe the given virtual address to see if there is memory or a
     * register actually there that responds to an access of the given 
     * size in bytes. This means using an OS function that probes without
     * causing a fatal bus error exception. Returns error if access failed
     */
    void (*old_berr)();
    void (*old_segv)();
    IIO_STATUS stat = iio_status_ok; 

    volatile uint64_t dummy;
    volatile uint8_t *uint8 = vaddr;
    volatile uint16_t *uint16 = vaddr;
    volatile uint32_t *uint32 = vaddr;
    volatile uint64_t *uint64 = vaddr;

    /* establish bus error/segmentation fault handler */
    old_berr = signal(SIGBUS, iio_probe_handle);
    old_segv = signal(SIGSEGV, iio_probe_handle);

    if ((stat = setjmp(iio_probe_jmpbuf)) == 0) {

	/* try reading */
	if (ptype & iio_ptype_read)
	    switch (bytes) {
	    case iio_size_8: dummy = *uint8;  break;
	    case iio_size_16: dummy = *uint16; break;
	    case iio_size_32: dummy = *uint32; break;
	    case iio_size_64: dummy = *uint64; break;
	    default:
		stat = iio_fatal("Silly bus cycle width");
	    }

	/* try writing */
	if (ptype & iio_ptype_write)
	    switch (bytes) {
	    case iio_size_8: *uint8  = dummy; break;
	    case iio_size_16: *uint16 = dummy; break;
	    case iio_size_32: *uint32 = dummy; break;
	    case iio_size_64: *uint64 = dummy; break;
	    default:
		stat = iio_fatal("Silly bus cycle width");
	    }
    }

    /* restore old bus error/segmentation fault handler */
    signal(SIGBUS, old_berr);
    signal(SIGSEGV, old_segv);

    return stat;
}

/*
 * --------------------------------------------------------------- PORT ACCESS
 *
 * Define PORT_MESSAGES for tracing a set of port access, and use
 * iio_port_dump() to late print the debug messages. For normal operation,
 * #undef PORT_MESSAGES
 */

#define PORT_MESSAGES
#undef PORT_MESSAGES

IIO_STATUS iio_port_alloc(
    void *paddr, unsigned int psize,
    void **pactual, void **vaddr, unsigned int *vsize
) {
    /*
     * Map port space. It's called this for symmetry with iio_shmap_alloc(),
     * and in principle does the same thing for ports. For LynxOS we don't
     * actually need to do anything much; we just return the mapped segment 
     * base addresses and sizes exactly as requested
     */
#ifndef __i386__
    return iio_error("Can't access ports on non-x86 processor");
#endif
    if (geteuid())
	return iio_error("Port access permission denied");

    *vaddr = *pactual = paddr;
    *vsize = psize;
    return iio_status_ok;
}


#ifdef PORT_MESSAGES
HIDDEN struct {
    unsigned addr, read, data, width;
} iio_port_messages[1000];
unsigned iio_port_mcount = 0;

#define PORT_MESSAGE(a, r, d, w) \
	iio_port_messages[iio_port_mcount].addr = (unsigned)(a); \
	iio_port_messages[iio_port_mcount].read = (r); \
	iio_port_messages[iio_port_mcount].data = (unsigned)(d); \
	iio_port_messages[iio_port_mcount].width = (w); \
	iio_port_mcount = (iio_port_mcount + 1) % 1000;

IIO_STATUS iio_port_dump(void) {
    /*
     * Print out the set of port messages accumulated since the last call
     * to this function
     */
    int count;
    for (count = 0; count < iio_port_mcount; ++count)
	printf(
	    "0x%04x %s 0x%0*x\n",
	    iio_port_messages[count].addr,
	    iio_port_messages[count].read ? "-->" : "<--",
	    iio_port_messages[count].width,
	    iio_port_messages[count].data
	);
    iio_port_mcount = 0;
    return iio_status_ok;
}

#else
#define PORT_MESSAGE(a, r, d, w)
#endif


uint8_t iio_port_get8(volatile uint8_t *addr) {
    /*
     * Read byte port and return value
     */
    uint8_t val;
#ifdef __i386__
    val = inb((unsigned)addr);
#else
    val = *addr;
#endif
    PORT_MESSAGE(addr, 1, val, 2);
    return val;
}


uint16_t iio_port_get16(volatile uint16_t *addr) {
    /*
     * Read word port and return value
     */
    uint8_t val;
#ifdef __i386__
    val = inw((unsigned)addr);
#else
    val = *addr;
#endif
    PORT_MESSAGE(addr, 1, val, 2);
    return val;
}


uint32_t iio_port_get32(volatile uint32_t *addr) {
    /*
     * Read long port and return value
     */
    uint8_t val;
#ifdef __i386__
    val = inl((unsigned)addr);
#else
    val = *addr;
#endif
    PORT_MESSAGE(addr, 1, val, 2);
    return val;
}


void iio_port_set8(volatile uint8_t *addr, uint8_t val) {
    /*
     * This is for writing byte port registers
     */
    PORT_MESSAGE(addr, 0, val, 2);
#ifdef __i386__
    outb((unsigned)addr, (unsigned)val);
#else
    *addr = val;
#endif
}


void iio_port_set16(volatile uint16_t *addr, uint16_t val) {
    /*
     * This is for writing byte port registers
     */
    PORT_MESSAGE(addr, 0, val, 4);
#ifdef __i386__
    outw((unsigned)addr, (unsigned)val);
#else
    *addr = val;
#endif
}


void iio_port_set32(volatile uint32_t *addr, uint32_t val) {
    /*
     * This is for writing long port registers
     */
    PORT_MESSAGE(addr, 0, val, 8);
#ifdef __i386__
    outl((unsigned)addr, (unsigned)val);
#else
    *addr = val;
#endif
}


/*
 * -------------------------------------------------------- OS-SPECIFIC CLEANUP
 */

IIO_STATUS iio_osdone(void) {
	/*
	 * This is called by iio_done() during the cleanup phase.
	 * The IIO data structures are removed using iio_done_iio(), and the
	 * mopping up is done here.
	 *
	 * On Lynxos we must attempt to remove the shared memory strutures 
	 * we have created, so they done hang around and get in the way of
	 * the next run. If they are still in use be another process, it does
	 * not matter; the remove will fail harmlessly
	 *
	 * Again we use the configuration file as a mutex lock, in case the
	 * cleanup of one IIO process hits the startup of another
	 */
	IIO_FILE file = -1;
	/* open the configuration file */
	iio_eret( iio_file_open("/etc/iio.conf", iio_fattr_rdonly, &file) );

#if 0
	printf("WARNING: have removed a flock in iio os/linux.c as it hangs\n");
#else
	/* obtain an exclusive lock on it */
	if (flock(file, LOCK_EX) != 0) 
		return iio_error(NULL);
#endif
	/* dismantle IIO data structures */
	iio_eret( iio_done_iio() );

	/* clean up shared memory thingys */
	iio_eret( iio_shmap_done() );
	iio_eret( iio_shmem_done() );

	/* release the lock */
	if (flock(file, LOCK_UN) != 0) 
		return iio_error(NULL);

	/* and close the file */
	iio_eret( iio_file_close(file) );

	return iio_status_ok;
}
