/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: opinfo.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * opinfo.c -- operation information table 
 * Robin Kirkham, June 1996
 */
#include "internal.h"

    /* this much match the IIO_OP enumerative */
IIO_OPINFO iio_opinfo[] = {
    { "nop",			"",	iio_oparg_none			 },
    { "show",			"",	iio_oparg_none			 },
    { "read",			"-->",	iio_oparg_out	| iio_oparg_data },
    { "readback",		"|->",	iio_oparg_out	| iio_oparg_data },
    { "write",			"<--",	iio_oparg_in	| iio_oparg_data },
    { "clear",			"<-0",	iio_oparg_none },

    { "space-io",		"<->",	iio_oparg_inout	| iio_oparg_addr },
    { "space-id",		"<->",	iio_oparg_inout	| iio_oparg_addr },
    { "space-int",		"<->",	iio_oparg_inout	| iio_oparg_addr },
    { "space-mem",		"<->",	iio_oparg_inout	| iio_oparg_addr },
    { "space-mem16",		"<->",	iio_oparg_inout	| iio_oparg_addr },
    { "space-mem24",		"<->",	iio_oparg_inout	| iio_oparg_addr },
    { "space-mem32",		"<->",	iio_oparg_inout	| iio_oparg_addr },

    { "sc-start",		"",	iio_oparg_none },
    { "sc-stop",		"",	iio_oparg_none },
    { "sc-free",		"",	iio_oparg_none },
    { "sc-read-current",	"-->",	iio_oparg_out	| iio_oparg_data },
    { "sc-read-target",		"-->",	iio_oparg_out	| iio_oparg_data },
    { "sc-write-current",	"<--",	iio_oparg_in	| iio_oparg_data },
    { "sc-write-target",	"<--",	iio_oparg_in	| iio_oparg_data },
    { "sc-write-target-dt",	"<--",	iio_oparg_in	| iio_oparg_data },
    { "sc-write-target-ddt",	"<--",	iio_oparg_in	| iio_oparg_data },
    { "sc-read-index",		"-->",	iio_oparg_out	| iio_oparg_data },
    { "sc-read-gain-p",		"-->",	iio_oparg_out	| iio_oparg_data },
    { "sc-read-gain-d",		"-->",	iio_oparg_out	| iio_oparg_data },
    { "sc-read-gain-i",		"-->",	iio_oparg_out	| iio_oparg_data },
    { "sc-write-gain-p",	"<--",	iio_oparg_in	| iio_oparg_data },
    { "sc-write-gain-d",	"<--",	iio_oparg_in	| iio_oparg_data },
    { "sc-write-gain-i",	"<--",	iio_oparg_in	| iio_oparg_data },

    { "adam-message",		"<->",	iio_oparg_inout	| iio_oparg_addr },

    { NULL, NULL, 0 }	/* NULL must be last */
};

IIO_STATUS iio_opinfo_lookup(char *string, IIO_OP *operation) {
    /*
     * Look up a string in iio_opinfo[], and return the operation code
     * for it, if found
     */
    IIO_OP op;

    /* check arguments */
    if (!string || !*string)
	return iio_fatal("NULL or empty operation name");
    if (! operation)
	return iio_fatal("NULL operation return pointer");

    for (op = 0; iio_opinfo[op].name; ++op)
	if (iio_string_cmp(string, iio_opinfo[op].name) == 0) {
	    *operation = op;
	    return iio_status_ok;
	}
    return iio_error("No such operation code");
}


IIO_STATUS iio_opinfo_show(void) {
    /*
     * Prints a table of current operation codes
     */
    IIO_OP op;

    iio_log(
	"    %2s  %-25.25s  %s\n",
	"",
	"name",
	"type"
    );

    for (op = 0; op < IIO_NOPS; ++op) {
	iio_log(
	    "    %02d  %-25.25s  %s%s%s%s\n",
	    (int)op,
	    iio_opinfo[op].name,
	    (iio_opinfo[op].arg & iio_oparg_in) ? "in " : "",
	    (iio_opinfo[op].arg & iio_oparg_out) ? "out " : "",
	    (iio_opinfo[op].arg & iio_oparg_data) ? "data " : "",
	    (iio_opinfo[op].arg & iio_oparg_addr) ? "addr " : ""
	);
    }

    return iio_status_ok;
}
