/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: operate.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * operate.c -- user operation functions
 * Robin Kirkham, June 1996
 */
#include "internal.h"


HIDDEN IIO_STATUS iio_operate_bitfield(
    IIO_OPEN *chan, IIO_OP op, IIO_UDATA udata, void *base
) {
    /*
     * BITFIELD OPERATIONS
     *
     * These are a bit different from normal operations, as the field
     * is represented by a channel range, but always a single user data
     * element. Hence, we call the passed get/set() functions directly
     * to access the user data bitfield.
     *
     * Bitfield channels are automatically registered (by iio_chnode() and
     * iio_bitfield_chnode()) in such a way that the channel first, number
     * and opnode->index etc. refer to bits in the user data bitfield,
     * rather than the normal channel numbers and offsets into a user data
     * array. This is at once cunning and extremly confusing.
     *
     * Once we have the data as an unsigned, we can call the "real" operate
     * function for each opnode. We use different get/set() functions
     * that simply copy data. 
     */
    IIO_OPNODE *opnode;
    IIO_STATUS stat = iio_status_ok;
    unsigned int data = 0;


    for (opnode = chan->opnode; opnode; opnode = opnode->next) {
	unsigned int droll = opnode->first + opnode->index ;
	unsigned int croll = opnode->first;
	unsigned int cmask = ((1 << opnode->number) - 1) << croll;
	unsigned int cdata = 0;


	/* take the module mutex */
	iio_shmutex_grab(opnode->chnode->module->mutex);

	/* for the first one only, get inward real user data, if required */
	if ((op == iio_op_write) && (opnode == chan->opnode)) {
	    opnode->udata = udata;
	    opnode->base = base;
	    data = (unsigned)iio_data_get(chan->opnode, -chan->opnode->index);
	}

	/* make the module driver get/set functions refer to cdata */
	opnode->udata = iio_udata_bit;
	opnode->base = &cdata;

	switch (op) {
	case iio_op_read:
	case iio_op_readback:

	    /* read or readback the real channel */
	    stat = opnode->chnode->rchnode->operate(
		opnode->chnode->rchnode->module->state,
		opnode->chnode->rchnode->module->reg,
		opnode,
		op,
		opnode->chnode->rseqno, 1
	    );

	    /* mask, roll ond OR result into return data */
	    data |= (((cdata & cmask) >> croll) << droll);
	    break;


	case iio_op_write:
	    /*
	     * If we are writing less than the full width of the real
	     * channel, we need to read it back first, and mask and OR
	     * in the new data
	     */
	    if (opnode->number != opnode->chnode->rchnode->width) {

		stat = opnode->chnode->rchnode->operate(
		    opnode->chnode->rchnode->module->state,
		    opnode->chnode->rchnode->module->reg,
		    opnode,
		    iio_op_readback,
		    opnode->chnode->rseqno, 1
		);

		/* mask and OR the new data into cdata */
		cdata &= ~cmask;
		cdata |=  cmask & ((data >> droll) << croll);

	    } else 
		cdata = cmask & ((data >> droll));
		

	    /* write back into the real channel */
	    stat = opnode->chnode->rchnode->operate(
		opnode->chnode->rchnode->module->state,
		opnode->chnode->rchnode->module->reg,
		opnode,
		iio_op_write,
		opnode->chnode->rseqno, 1
	    );
	    break;

	default:
	    return iio_error("Operation code not supported for bitfield");
	}

	/* for the last one only, set outward real user data, if required */
	if ((op == iio_op_read || op == iio_op_readback) && (!opnode->next)) {
	    chan->opnode->udata = udata;
	    chan->opnode->base = base;
	    iio_data_set(chan->opnode, -chan->opnode->index, data);
	}

	/* release the module mutex */
	iio_shmutex_drop(opnode->chnode->module->mutex);
    }
    return stat;
}


HIDDEN IIO_STATUS iio_operate_call(
    IIO_OPEN *chan, IIO_OP op, IIO_UDATA udata, void *base
) {
    /*
     * Operate on a channel, using the given get/set user data functions
     * and the given base user data pointer. This is the interface point
     * with the driver operation functions; all operations go through 
     * this point. The user interface functions call this one with various
     * different functions for set() and get() functions, according to
     * the data type
     */
    IIO_STATUS stat = iio_status_ok;
    IIO_OPNODE *opnode;

    /* check arguments */
    if (! iio_state) 
	return iio_fatal("Library not initialised");
    if (!chan || chan->magic != iio_magic_open)
	return iio_fatal("NULL or bad channel pointer");
    if (!base)
	return iio_fatal("NULL channel data pointer");


    /*
     * If logging is on, print a header for the logging data that will
     * follow from the get/set() functions the operate function will call
     */
    if (chan->opnode->log)
	iio_log(
	    "%s: %s[%d]:\n",
	    chan->name,
	    iio_opinfo[op & IIO_SPACE_MASK].name,
	    chan->number
	);

    /* take channel mutex if there is one */
    if (chan->mutex) {
	iio_eret( iio_mutex_grab(chan->mutex) );
    }


    /* "normal" and bitfield operations are done differently */
    if (! chan->opnode->chnode->rchnode)
	/*
	 * NORMAL OPERATIONS
	 *
	 * Normally we just rip through the opnode list and call the 
	 * operation functions, passing the opnode itself, some important
	 * used things out of it (so the driver does not need to know the 
	 * internals of the opnode) and the get/set() functions, which were
	 * passed to us
	 */
	for (opnode = chan->opnode; opnode; opnode = opnode->next) {

	    /* take the module mutex */
	    iio_shmutex_grab(opnode->chnode->module->mutex);

	    /* set the user data type and base for the get/set funcitons */
	    opnode->udata = udata;
	    opnode->base = base;
	    opnode->op = op;

	    /* call the operate function of the driver for each channel */
	    stat = opnode->chnode->operate(
		opnode->chnode->module->state,
		opnode->chnode->module->reg,
		opnode,
		op,
		opnode->first,
		opnode->number
	    );

	    /* release the module mutex */
	    iio_shmutex_drop(opnode->chnode->module->mutex);
	}
    else
	stat = iio_operate_bitfield(chan, op, udata, base);

    /* drop the channel mutex if there is one */
    if (chan->mutex) {
	iio_eret( iio_mutex_drop(chan->mutex) );
    }

    return stat;
}


IIO_STATUS iio_operate(IIO chan, IIO_OP op, int data[]) {
    /*
     * Operate on a channel, passing a pointer to integer in/out data.
     * This is done by calling iio_operate_call() passing pointers to
     * get/set conversion functions that do no conversions at all
     */
    iio_eret( iio_operate_call(chan, op, iio_udata_int, (void *)data) );
    return iio_status_ok;
}


IIO_STATUS iio_operate_real(IIO chan, IIO_OP op, double data[]) {
    /*
     * Operate on a channel, passing a pointer to double in/out data.
     * This is done by calling iio_operate_call() passing pointers to
     * get/set conversion functions which convert the integer values
     * used in the driver to the user's double format data, using the 
     * scaling factors (or whatever) in the chinfo blocks
     */
    iio_eret( iio_operate_call(chan, op, iio_udata_real, (void *)data) );
    return iio_status_ok;
}


IIO_STATUS iio_operate_addr(IIO chan, IIO_OP op, void *addr[]) {
    /*
     * Operate on a channel, passing a pointer to address in/out data.
     * It could be argued that this function is not necessary, and that
     * iio_operate_int() would do just as well, since it only deals in 
     * pointers to objects anyway. But since we deal in arrays of the
     * objects, size is indeed important
     */
    iio_eret( iio_operate_call(chan, op, iio_udata_addr, (void *)addr) );
    return iio_status_ok;
}


IIO_STATUS iio_operate_in(IIO chan, IIO_OP op, int data) {
    /*
     * Operate on a simple channel, passing actual integer
     * inwards only data.
     */

    /* check that simple channels and inward integer data only */ 
    if (chan->number != 1)
	return iio_error("Wrong operate function used for channel");
    if (iio_opinfo[op].arg != (iio_oparg_in | iio_oparg_data))
	return iio_error("Wrong operate function used for operation");

    iio_eret( iio_operate_call(chan, op, iio_udata_int, (void *)&data) );
    return iio_status_ok;
}

IIO_STATUS iio_operate_inreal(IIO chan, IIO_OP op, double data) {
    /*
     * Operate on a simple channel, passing actual double
     * inwards only data.
     */

    /* check that simple channels and inward data only */ 
    if (chan->number != 1)
	return iio_error("Wrong operate function used for channel");
    if (iio_opinfo[op].arg != (iio_oparg_in | iio_oparg_data))
	return iio_error("Wrong operate function used for operation");

    iio_eret( iio_operate_call(chan, op, iio_udata_real, (void *)&data) );
    return iio_status_ok;
}

IIO_STATUS iio_operate_inaddr(IIO chan, IIO_OP op, void *addr) {
    /*
     * Operate on a simple channel, passing actual pointer
     * inwards only data.
     */

    /* check that simple channels and inward addr only */ 
    if (chan->number != 1)
	return iio_error("Wrong operate function used for channel");
    if (iio_opinfo[op].arg != (iio_oparg_in | iio_oparg_addr))
	return iio_error("Wrong operate function used for operation");

    iio_eret( iio_operate_call(chan, op, iio_udata_addr, (void *)&addr) );
    return iio_status_ok;
}
