/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: open.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * open.c -- open a channel
 * Robin Kirkham, June 1996
 */
#include <stdio.h>
#include "internal.h"


HIDDEN int iio_open_cmp(IIO_SLL *s1, IIO_SLL *s2) {
    /*
     * Used by the ordered singly-linked list function iio_sll_insert().
     * The open channel are ordered alphabetically by opening name
     * (which may be an alias)
     */
    return iio_string_cmp(((IIO_OPEN *)s1)->name, ((IIO_OPEN *)s2)->name);
}


HIDDEN IIO_STATUS iio_open_opnode(
    IIO_CHNODE *chnode, unsigned first, unsigned number, unsigned index,
    IIO_OPNODE **opnode
) {
    /*
     * Create a new open channel node structure, based on the information
     * in the given IIO_CHNODE block, but with the given first and number args
     */
    IIO_OPNODE *new;

    /* allocate a new opnode block */
    iio_eret( iio_mem_alloc(sizeof(IIO_OPNODE), (void **)&new) );

    /* fill it in */
    new->magic = iio_magic_opnode;
    new->chnode = chnode;
    new->first = first;
    new->number = number;
    new->index = index;
    new->log = iio_bool_false;

    /* return the pointer to it */
    *opnode = new;

    return iio_status_ok;
}


IIO_STATUS iio_open(char *name, IIO_OFLAG flags, IIO_OPEN **chan) {
    /*
     * Open a channel, or channel range, with the given name, which may
     * be of any of the many acceptable forms. The flags argument might
     * mean something one day, or it might get abolished. A pointer to
     * a dynamically allocated IIO_OPEN structure is returned into chan
     */
    IIO_NAMEX namex;
    IIO_CHNODE *chnode;
    IIO_OPEN *channel;
    IIO_OPNODE *top = NULL, *new = NULL, *prev = NULL;
    IIO_BOOL started, finished, log;
    unsigned index = 0, seqno;
    int which;
    
    /* check some arguments */
    if (!name || !*name)
	return iio_fatal("NULL or empty name string");
    if (!chan)
	return iio_fatal("NULL channel pointer");

    /*
     * Explode the name into its constituent parts, which
     * includes resolving any aliases used in the name
     */
    iio_eret( iio_namex(name, &namex) );

#if 0
    printf("nform = %d\n", namex.form);
    printf("mident = <%s>\n", namex.mident);
    printf("mseq = %d\n", namex.mseq);
    printf("ctype = %d <%s>\n", namex.type, iio_chtype_string[namex.type]);
    printf("cwidth = %d\n", namex.width);
    printf("cseq1 = %d\n", namex.seq1);
    printf("cseq2 = %d\n", namex.seq2);
#endif

    /*
     * Enforce the limit on bitfield range size, must be less than the 
     * number of bits in an unsigned int. This is about the only limit
     * there is
     */
    if ((
	    (namex.type == iio_chtype_bi)   ||
	    (namex.type == iio_chtype_bo)   ||
	    (namex.type == iio_chtype_bio)  ||
	    (namex.type == iio_chtype_boco) ||
	    (namex.type == iio_chtype_bocio)
	) && ((namex.seq2 - namex.seq1) >= channelwidth(unsigned int))
    )
	return iio_error("Bitfield range too large");


    /*
     * Opening a channel does not involve actually accessing the driver, but
     * getting the information together so accesses to the driver for real
     * operations can be done fairly eficiently. For each open channel at least
     * one IIO_OPNODE structure is allocated; if the channel is a range which
     * spans more than one module (or at least chnode group), there are other
     * IIO_OPNODE structures, linked into a secondary list hanging off an
     * IIO_OPEN structure, which represents the open channel.
     * 
     * So in essence, there is a IIO_OPNODE structure created for each
     * IIO_CHNODE structure. The trick is to determine which IIO_CHNODE's.
     * This is worked out by using the sequence number, and the form of the
     * channel name/range, by simply searching the chnode list for the modules
     * that fit.
     *
     * The index field is the index into the channel range of the first
     * simple channel referenced by each opnode. It tells the get/set() 
     * functions that the driver operate function calls where to put the
     * data as it is read/written.
     */

    /* state flags */
    started = finished = iio_bool_false;

    /*
     * Switch on logging if (a) the user wants it (in the call to iio_open()),
     * (b) any referenced module has logging enabled, or (c) any simple channel
     * has logging enabled. (b) and (c) are specified in the config file
     */
    log = (flags & iio_oflag_log);

    /* search the chnode list */
    for (chnode = iio_state->chnode; chnode; chnode = chnode->next) {
	/*
	 * Test if we are likely to be interested in this chnode--
	 * type must match, plus width and/or module, depending on
	 * name form. Note than if the width in a chnode block is zero,
	 * width does not matter with this channel.
	 *
	 * If no match, continue with next try
	 */
	if ((namex.type == chnode->type)) 
	    switch (namex.form) {
	    case iio_nform_gg:
		/* global generic form */
		break;

	    case iio_nform_gs:
		/* global specific form -- check width too */
		if (chnode->width == 0 || namex.width == chnode->width)
		    break;
		else
		    continue;

	    case iio_nform_lg:
		/* local generic form -- check module */
		if (namex.module == chnode->module)
		    break;
		else
		    continue;

	    case iio_nform_ls:
		/* local specific form -- check module AND width */
		if (
		    namex.module == chnode->module && (
			chnode->width == 0 ||
			namex.width == chnode->width
		    )
		)
		    break;
		else
		    continue;
	    }
	else
	    continue;

	/*
	 * Now we see if the channel range the chnode block offers intersects
	 * with the channel range we want, and in what way. We compare the 
	 * start and end sequence numbers of the range we want with the start
	 * and end sequence numbers (strictly, start and number) available in
	 * that chnode. There are sixteen possibilities, some impossible, some
	 * indicating we need to keep searching, some we need to give up, and
	 * four useful cases where the ranges overlap and we need to obtain a
	 * IIO_OPNODE block and fill it in. These are: (0) where the range we
	 * want is all on the one chnode block; (1) the range starts on this
	 * chnode, and continues on others; (2) the range completely encloses
	 * this chnode; and (3) the range finishes on this chnode. These cases
	 * are decoded by the ghastly expression below:
	 */
	which = (
	    ((namex.seq1 >= chnode->seqno[namex.form]) ? 1 : 0) |
	    ((namex.seq1 < chnode->seqno[namex.form]+chnode->number) ? 2 : 0) |
	    ((namex.seq2 >= chnode->seqno[namex.form]) ? 4 : 0) |
	    ((namex.seq2 < chnode->seqno[namex.form]+chnode->number) ? 8 : 0)
	);

	/*
	 * For each case, we obtain a new IIO_OPNODE block. The first such
	 * block is called the top block, and is the one later linked into the 
	 * main open channel list (subsequent blocks for this open channel
	 * range are linked to this by the `more' field, not `next').
	 *
	 * The local specific channel sequence number corresponding to the
	 * first in the range we want (or part thereof) and the number
	 * need to be computed and put in the block.
	 */
	new = NULL;

	switch (which) {
	case 0x5:
	    /* we have't reached a useful seqno yet: keep searching */
	    continue;

	case 0xa:
	    /* we seem to have gone past useful seqnos: something is wrong */
	    return iio_fatal("Searched too far?");

	case 0xf:
	    /* the range we want is all on the one chnode */
	    index = -(namex.seq1 - chnode->seqno[namex.form]);
	    iio_eret(
		iio_open_opnode(
		    chnode,
		    namex.seq1 - chnode->seqno[namex.form],
		    namex.seq2 - namex.seq1 + 1,
		    index,
		    &new
		)
	    );
	    started = finished = iio_bool_true;
	    break;

	case 0x7:
	    /* the range we want starts on this chnode */
	    index = -(namex.seq1 - chnode->seqno[namex.form]);
	    iio_eret(
		iio_open_opnode(
		    chnode,
		    namex.seq1 - chnode->seqno[namex.form],
		    chnode->number - (namex.seq1 - chnode->seqno[namex.form]),
		    index,
		    &new
		)
	    );
	    index += chnode->number;
	    started = iio_bool_true;
	    break;

	case 0x6:
	    /* this chnode is completely used */
	    iio_eret(
		iio_open_opnode(
		    chnode,
		    0,
		    chnode->number,
		    index,
		    &new
		)
	    );
	    index += chnode->number;
	    break;

	case 0xe:
	    /* the range we want ends on this chnode */
	    iio_eret(
		iio_open_opnode(
		    chnode,
		    0,
		    namex.seq2 - chnode->seqno[namex.form] + 1,
		    index,
		    &new
		)
	    );
	    finished = iio_bool_true;
	    break;

	default:
	    return iio_fatal("Impossible which case");
	}

	/* 
	 * Check each chinfo in the chnode for the logging flag, then
	 * check the module block as well
	 */
	for (seqno = new->first; seqno < new->first + new->number; ++seqno)
	   log |= chnode->chinfo[seqno].log;
	log |= chnode->module->log;

	/*
	 * Link new into a sub-list for this open channel.
	 * top is the head of this list
	 */
	if (new) {
	    if (!top)
		top = new;
	    if (prev) 
		prev->next = new;
	    prev = new;
	}
	 
	/* see if we are finished */
	if (started && finished)
	    break;
    }

    /* check if we started and finished properly */
    if (!started || !finished) 
	return iio_error("No such channel or range");

    /* create a new IIO_OPEN structure for the channel */
    iio_eret( iio_mem_alloc(sizeof(IIO_OPEN), (void **)&channel) );

    /* duplicate the name it was opened with */
    iio_eret( iio_string_dup(name, &channel->name) );
    channel->magic = iio_magic_open;
    channel->opnode = top;
    channel->number = namex.seq2 - namex.seq1 + 1;

    if (log) {
	/*
	 * Logically, the logging status should be in the open channel
	 * descriptor IIO_OPEN, rather than the IIO_OPNODE blocks, but it is
	 * the opnodes that get passed to the module operate functions and the
	 * data get/set() functions, which actually do most of the logging.
	 * So we have to duplicate the flag in each of the blocks
	 */
	IIO_OPNODE *opnode;

	for (opnode = top; opnode; opnode = opnode->next)
	    opnode->log = log;
    }

    /*
     * Create a mutex for this open channel, but only if the channel has more
     * than one opnode. This there is only one opnode, the module mutex is
     * sufficient. This is a process-wide mutex (on LynxOS)
     */
    if (channel->opnode->next) {
	iio_eret( iio_mutex_create(&channel->mutex) );
    }

    /* take the process-wide open-channel list mutex */
    iio_eret( iio_mutex_grab(iio_state->omutex) );

    /* link the top channel into the open channel list */
    iio_eret(
	iio_sll_insert(
	    (IIO_SLL **)&iio_state->open,
	    (IIO_SLL *)channel,
	    iio_open_cmp
	)
    );

    /* release the process-wide open-channel list mutex */
    iio_eret( iio_mutex_drop(iio_state->omutex) );

    /* return the channel */
    *chan = channel;

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_open_show_one(IIO_OPEN *chan) {
    /*
     * TEMPORARY FUNCTION
     * Prints some stuff about one channel only.
     * Called only by iio_show()
     */
    IIO_OPNODE *opnode;

    if (!chan || chan->magic != iio_magic_open)
	return iio_fatal("NULL or bad channel pointer");

    printf("%s:\n", chan->name);
    for (opnode = chan->opnode; opnode; opnode = opnode->next) 
	printf(
	    "    %s.%d:%s%d.%d-%d  [%d] %d\n",
	    opnode->chnode->module->minfo->ident,
	    opnode->chnode->module->seq,
	    iio_chtype_string[opnode->chnode->type],
	    opnode->chnode->width,
	    opnode->first,
	    opnode->first + opnode->number - 1,	/* last */
	    opnode->index,
	    opnode->chnode->number
	);
	
    return iio_status_ok;
}


IIO_STATUS iio_open_show(IIO_OPEN *chan) {
    /*
     * TEMPORARY FUNCTION
     * Prints some stuff about the given channel, or if it is NULL,
     * does so for all the channels in the open channel list
     */
    if (chan)
	return iio_open_show_one(chan);

    for (chan = iio_state->open; chan; chan = chan->next)
	iio_fret( iio_open_show_one(chan) );
    return iio_status_ok;
}
