/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: namex.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * namex.c -- channel name expansion function
 * Robin Kirkham, June 1996
 */
#include "internal.h"


HIDDEN IIO_STATUS iio_namex_chan(char *pchan, IIO_NAMEX *namex) {
    /*
     * Breaks the channel part of a channel name (which might the entire
     * name, or the part after the :) into the channel type, width, and
     * sequence numbers:
     *
     *		<type>.<seq1>
     *		<type>.<seq1>-<seq2>
     *		<type><width>.<seq1>
     *		<type><width>.<seq1>-<seq2>
     *
     * namex->form is amended from lg to ls, or gg to gs form, if a width
     * is found in the name
     */
    char *pseq1, *pseq2;
    long seq1, seq2;

    /* look for a . then sequence number, or sequence number range */
    if (! (pseq1 = iio_string_pbrk(pchan, "."))) 
	return iio_error("Missing '.' in channel name");

    /* get the sequence number, or numbers */
    *pseq1 = '\0';
    ++pseq1;

    if ((pseq2 = iio_string_pbrk(pseq1, "-"))) {
	/* it's a range */
	*pseq2 = '\0';
	++pseq2;
    } else
	pseq2 = pseq1;

    /* decode the two sequence numbers */
    iio_eret( iio_decode_long(pseq1, &seq1) );
    iio_eret( iio_decode_long(pseq2, &seq2) );

    /* sequence numbers have to be positive */
    if (seq1 < 0 || seq2 < 0)
	return iio_error("Can't have negative sequence numbers");
    if (seq2 < seq1)
	return iio_error("Sequence number in range must be in order");
    namex->seq2 = (unsigned)seq2;
    namex->seq1 = (unsigned)seq1;

    /* decode channel type/width part */
    iio_eret( iio_chtype_find(pchan, &(namex->type), &(namex->width)) );

    /* if width was specified, adjust channel name type */
    if (namex->width) {
	/* it is a specific form (gs or ls) */
	if (namex->form == iio_nform_gg)
	    namex->form = iio_nform_gs;
	if (namex->form == iio_nform_lg)
	    namex->form = iio_nform_ls;
    }

    return iio_status_ok;
}


IIO_STATUS iio_namex(char *name, IIO_NAMEX *namex) {
    /*
     * This takes a channel name in any form and explodes it into its
     * constituent parts, which are written into the CALLER SUPPLIED 
     * namex structure. Any aliases in the name are resolved on the alias
     * list. The namex structure contains the a module ident string pointer,
     * an module sequence number, the channel type, width and channel sequence
     * numbers, and finally the form of the name, i.e. gg, gs, lg, or ls:
     *
     *	<type>.<seq>					gg form
     *	<type>.<seq1>-<seq2>				gg form (range)
     *	<type><width>.<seq>				gs form
     *	<type><width>.<seq1>-<seq2>			gs form (range)
     *	<mident>.<mseq>:<type>.<seq>			lg form
     *	<mident>.<mseq>:<type>.<seq1>-<seq2>		lg form (range)
     *	<mident>.<mseq>:<type><width>.<seq>		ls form
     *	<mident>.<mseq>:<type><width>.<seq1>-<seq2>	ls form (range)
     *
     * where <type> is a channel type string; <seq> is a channel sequence 
     * number, <seq1>-<seq2> is a channel sequence number range (seq2 > seq1);
     * <width> is the channel width in bits; <mident> is a valid module ident
     * code (at least in minfo list); <mseq> is a module sequence number.
     * For non-range forms, seq1 and seq2 in the namex structure are equal.
     */
    char buffer[200], *pchan = buffer, *pmod = buffer;
    IIO_ATYPE atype;

    /* check some parameters */
    if (!name || !*name)
	return iio_fatal("NULL or empty channel name string");
    if (!namex)
	return iio_fatal("NULL namex pointer");

    /* set magic number to something silly */
    namex->magic = (IIO_MAGIC)(-1);

    /*
     * First, resolve name as a global alias. If it actually happens to be
     * a global alias, we should then have a pointer to a channel name of some
     * form. If it is already a proper channel name, we've only wasted time
     */
    iio_fret( iio_alias_find(name, &atype, &name) );
    if (atype == iio_atype_module || atype == iio_atype_local)
	return iio_error("Can't use a module or local alias here");

    /* duplicate, so we can parse destructively */
    if (iio_string_len(name) > sizeof(buffer) - 1)
	return iio_error("Channel name too long");
    iio_string_cpy(buffer, name);


    /* look for a dividing : */
    if ((pchan = iio_string_pbrk(buffer, ":"))) {

	/* name is in lg or ls form (module:channel): split into two */
	*pchan++ = '\0';

	/* amended to ls form if necessary by iio_namex_chan() */
	namex->form = iio_nform_lg;

	/*
	 * Process module part first: look up the string in the alias
	 * list as a module alias, then call iio_module_find() to see
	 * if it's in the installed module list
	 */

	iio_fret( iio_alias_find(pmod, &atype, &pmod) );
	if (atype == iio_atype_global || atype == iio_atype_local)
	    return iio_error("Can't use a global or local alias here");

	iio_eret( iio_module_find(pmod, &(namex->module)) );

	/*
	 * Process the channel part, by looking it up as a local alias,
	 * duplicating the result again, and splitting it up into type,
	 * width, seqnos and so on
	 */
	iio_fret( iio_alias_find(pchan, &atype, &pchan) );
	if (atype == iio_atype_global || atype == iio_atype_module)
	    return iio_error("Can't use a global or module alias here");

	if (iio_string_len(pchan) > sizeof(buffer) - 1)
	    return iio_error("Channel name too long");
	iio_string_cpy(buffer, pchan);

	iio_eret( iio_namex_chan(buffer, namex) );

    } else {

	/* name is in gg or gs form (no module part) */
	namex->module = NULL;

	/* amended to gs form if necessary by iio_namex_chan() */
	namex->form = iio_nform_gg;

	/* explode channel part */
	iio_eret( iio_namex_chan(buffer, namex) );
    }

    /* if we get here, success! */
    namex->magic = iio_magic_namex;
    return iio_status_ok;
}
