/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: vmivme4100.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * vmivme4100.c -- driver for VMIC VMIVME-4100 VME Digital IO Module
 *
 * The VMIC VMIVME-4100 provides 16 12-bit DAC channels on the VMEbus.
 *
 *	-address <addr>			VME short base address
 *	-bus <channel>			VME bus (if not vme.0)
 *
 * This driver is essentially complete, but is NOT TESTED.
 *
 * We assume that the board has been set up for update under program control
 * (jumpers JC1 to JC4 are IN), rather than immediate mode. This permits the
 * driver to load a range of channels into the DAC preload registers, then
 * promulgate them all at once, for simultaneous update.
 *
 * If the jumpers are out, I expect the individual output will update
 * immediately, and the attempted promulgate will do nothing. Similarly,
 * if the jumpers are in, and the external update jumper (JC5) is in as well,
 * the promulgate will do nothing and promulgation will be caused by a
 * low-pulse inserted into the external trigger input.
 *
 * Therefore, there are no driver options for these two setups.
 */
#include "../internal.h"

    /* factory default base address (VME A16). This is a silly choice */
#define BASE		0x0000

    /* amount of address space occluded (bytes) */
#define SIZE		0x20

    /* the number of channels */
#define NCHAN		16

    /* CSR register bit definitions */
#define CSR_OEP3	(1 <<  8)	/* enable outputs to P3 connector */
#define CSR_PROM	(1 <<  9)	/* promulgate data to DACS */
#define CSR_ATOTB	(1 << 10)	/* clock address bits to test bus */
#define CSR_OTOTB	(1 << 11)	/* switch outputs to test bus */
#define CSR_TB1		(1 << 12)	/* switch to P2 test bus 1 */
#define CSR_TB2		(1 << 13)	/* switch to P2 test bus 2 */
#define CSR_FAIL	(1 << 14)	/* switch off fail lamp */
#define CSR_CSR		(1 << 15)	/* must be asserted to write to CSR */


    /* register pointer structure */
struct IIO_MREG {
    volatile uint16_t *csr;		/* control/status register */
    volatile uint16_t *dr[16];	/* data registers */
};

    /* state data structure */
struct IIO_MSTATE {
    volatile uint16_t dr[16];	/* data register shadows */
};


HIDDEN IIO_STATUS iio_vmivme4100_dio(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * The VMIC2534A can operate as a dio or di/do device, depending on 
     * whether the readback jumpers are connected or not. Since it's just
     * too complicated to support every combination, we say either all of
     * these are on, or they are off
     */
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_readback:
	    iio_eret(
		iio_data_set(opnode, seqno, (int)(state->dr[seqno]) - 0x800)
	    );
	    break;

	case iio_op_write:
	    (*reg->dr[seqno]) = state->dr[seqno] =
		(uint16_t)(iio_data_get(opnode, seqno) + 0x800) & 0xfff;
	    break;

	case iio_op_clear:
	    (*reg->dr[seqno]) = state->dr[seqno] = 0x800;
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}

    /* promulgate through to DACS */
    if (op == iio_op_write || op == iio_op_clear)
	*reg->csr = CSR_OEP3 | CSR_PROM | CSR_FAIL | CSR_CSR;

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_vmivme4100_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialise the module using the configuration already loaded in the
     * state and register structures. First probe the addresses and see if
     * something is at least there. Then initialise the registers to match
     * the state (or vice-versa).
     *
     * Beacuse there is no read-back from the device, we have to clear
     * the outputs and the output shadows
     */
    int count;

    /* probe the register addresses */
    iio_eret( iio_probe(reg->csr, iio_size_16, iio_ptype_read) ); 

    /* built in test? you'd have to be lucky */

    /* clear the outputs and shadows */
    for (count = 0; count < NCHAN; ++count)
	*reg->dr[count] = state->dr[count] = 0x800;

    /* clear the front-panel fail lamp, enable the output transistors */
    *reg->csr = CSR_OEP3 | CSR_PROM | CSR_FAIL | CSR_CSR;

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_vmivme4100_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    uint16_t base = BASE;
    IIO_MREG *reg;
    IIO_CHNODE *chnode;
    IIO chan = NULL;
    int count;


    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* get the arguments */
    iio_eret( iio_arg(argv, "address", iio_arg_uint16, &base) );
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &chan) );

    /* this board always uses D16 short VME addressing */
    if (!chan)
	iio_eret( iio_open("vme.0", 0, &chan) );

    /* register the channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_dac, 12, NCHAN,
	    iio_vmivme4100_dio, &chnode
	)
    );

    /* map a window into memory */
    iio_eret( iio_map(chan, iio_space_mem16, base, SIZE) );


    /* for each DAC channel */
    for (count = 0; count < NCHAN; ++count) {

	/* resolve data register addresses into register structure */
	iio_eret(
	    iio_resolve(
		chan, iio_space_mem16, iio_size_16,
		base + (count * 2), (void **)&reg->dr[0]
	    )
	);

	/* enter channel scale factor */
	iio_eret(
	    iio_chnode_linear(
		chnode, count,
		20.0/4096, 0.0, "V"
	    )
	);
    }

    /* the CSR address is the same as dr[0] */
    reg->csr = reg->dr[0];

    iio_close(chan);
    return iio_status_ok;
}


IIO_STATUS iio_vmivme4100(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"vmivme4100",
	"VMIC VMIVME-4100 16-chan DAC",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_vmivme4100_install,
	iio_vmivme4100_init
    );
}
