/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: vmivme2532a.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * vmivme2532a.c -- driver for VMIC Digital IO Modules 
 *		    VMIVME-2532A and VMIVME-2534
 *
 * The VMIC VMIVME-2532A and VMIVME-2534 each provide two 16-bit high-voltage
 * OCIO channels on the VMEbus. I believe the only difference between them
 * is the connector arrangement (front panel on the 2532A and via P2 on the
 * 2534). It is possible the factory default base address is different
 *
 *	-address <addr>			VME short base address
 *	-bus <channel>			VME bus (if not vme.0)
 *
 * This driver needs a bit of work.
 */
#include "../internal.h"

    /* factory default base address (VME A16) */
#define BASE		0xff60

    /* amount of address space occluded (bytes) */
#define SIZE		8

    /* 16-bit register addresses relative base */
#define REG_DR0		0x00		/* bits 31-16 */
#define REG_DR1		0x02		/* bite 15-0 */
#define REG_CSR		0x04		/* control/status register */

    /* CSR register bit definitions */
#define CSR_TM1		(1 << 7)	/* test mode bit 1 */
#define CSR_TM2		(1 << 6)	/* test mode bit 2 */
#define CSR_FAIL	(1 << 5)	/* front panel fail lamp */


    /* register pointer structure */
struct IIO_MREG {
    volatile uint16_t *dr[2];	/* data registers */
    volatile uint16_t *csr;		/* control/status register */
    int ocio;				/* OCIO mode flag */
    int invert;				/* data output invert flag */
};

    /* state data structure */
struct IIO_MSTATE {
    volatile uint16_t dr[2];	/* data register readback */
};


HIDDEN IIO_STATUS iio_vmivme2532a_ocio(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * The VMIC2534A can operate as a ocio or di/oco device, depending on 
     * whether the readback jumpers are connected or not. Since it's just
     * too complicated to support every combination, we say either all of
     * these are on, or they are off
     */
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, (int)(*reg->dr[seqno])) );
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, (int)(state->dr[seqno])) );
	    break;

	case iio_op_write:
	    (*reg->dr[seqno]) = state->dr[seqno] =
		(uint16_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_clear:
	    (*reg->dr[seqno]) = state->dr[seqno] = 0;
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_vmivme2532a_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialise the module using the configuration already loaded in the
     * state and register structures. First probe the addresses and see if
     * something is at least there. Then initialise the registers to match
     * the state (or vice-versa).
     *
     * Beacuse there is no read-back from the device, we have to clear
     * the outputs and the output shadows
     */

    /* probe the register addresses */
    iio_eret( iio_probe(reg->dr[0], iio_size_16, iio_ptype_read) ); 

    /* built in test? you'd have to be lucky */

    /* clear the outputs and shadows */
    *reg->dr[0] = state->dr[0] = 0;
    *reg->dr[1] = state->dr[1] = 0;

    /* clear the front-panel fail lamp, enable the output transistors */
    *reg->csr = CSR_TM1 | CSR_TM2 | CSR_FAIL;

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_vmivme2532a_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    uint16_t base = BASE;
    IIO_MREG *reg;
    IIO chan = NULL;


    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* get the arguments */
    iio_eret( iio_arg(argv, "address", iio_arg_uint16, &base) );
    iio_eret( iio_arg(argv, "invert", iio_arg_bool, &reg->invert) );
    iio_eret( iio_arg(argv, "ocio", iio_arg_bool, &reg->ocio) );
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &chan) );

    /* this board always uses D16 short VME addressing */
    if (!chan)
	iio_eret( iio_open("vme.0", 0, &chan) );

    /* map a window into memory */
    iio_eret( iio_map(chan, iio_space_mem16, base, SIZE) );


    /* initialise register pointer structure */
    iio_eret(
	iio_resolve_list(
	    chan, iio_space_mem16,
	    iio_size_16, base + REG_DR1, &reg->dr[0],
	    iio_size_16, base + REG_DR0, &reg->dr[1],
	    iio_size_16, base + REG_CSR, &reg->csr,
	    0
	) 
    );
    iio_close(chan);

    /* register the channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_ocio, 16, 2,
	    iio_vmivme2532a_ocio, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_vmivme2532a(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "vmivme2532a",
	    "VMIC VMIVME-2532A Digital IO",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_vmivme2532a_install,
	    iio_vmivme2532a_init
	)
    );
    iio_eret(
	iio_minfo(
	    "vmivme2534",
	    "VMIC VMIVME-2534 Digital IO",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_vmivme2532a_install,
	    iio_vmivme2532a_init
	)
    );
    return iio_status_ok;
}
