/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: vipc610.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * vipc610.c -- driver for GreenSpring VIPC610 VME IP Carrier
 *
 * The VIPC610 and VIPC616 are VMEbus IP carriers providing four IP slot
 * addressed from the VMEbus. The VIPC616 is an updated model with more
 * features.
 *
 *	-address <addr>			VME base address
 *	-memory/-no-memory 		map IP memory space to bus
 *	-bus <channel> 			bus (if not vme.0)
 *
 * At present the driver uses the VIPC616 at 610 level; memory mapping is
 * not done, and more work is needed on this one...
 */
#include "../internal.h"


    /* factory default base address (VME A16) */
#define BASE 0x6000

    /* amount of A16 and A24 address space occluded (bytes) */
#define A16SIZE 0x800			/* 1024 bytes */
#define A24SIZE 0x80000			/* 256k bytes */


    /* register pointer structure */
struct IIO_MREG {
    IIO bus;				/* channel for the VME bus */
    uint16_t a16addr;		/* VME A16 base address (IP IO/ID) */
    uint32_t a24addr;		/* VME A24 base address (IP memory) */
    IIO_BOOL memory;			/* memory space is mapped to VME */
};


HIDDEN IIO_STATUS iio_vipc610_ip(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * An IP carrier has to support operations to resolve IP local IO, ID, 
     * MEM and INT addresses to virtual. This is done by computing the VME
     * A16 or A24 address of the requested local location, then resolving that
     * to a processor physical address by calling the resolve operation on the
     * channel representing the VME bus.
     *
     * Note that we don't call iio_resolve() here, as that function will do
     * a physical-to-virtual lookup as well, which we don't want at this
     * stage. 
     */
    unsigned slot;

    for (slot = first; slot < first + number; ++slot) {
	uint32_t offs = (uint32_t)iio_data_get_addr(opnode, slot);
	IIO_SIZE size = (IIO_SIZE)(op & IIO_SIZE_MASK);
	void *result;

	switch (op & IIO_SPACE_MASK) {

	case iio_space_io:
	    if (offs > 0x7f)
		return iio_error("IPIO address out of range");
	    result = (void *)(reg->a16addr + 0x100 * slot + 0x00 + offs);
	    iio_eret(
		iio_operate_addr(reg->bus, iio_space_mem16 | size, &result) 
	    );
	    iio_eret( iio_data_set_addr(opnode, slot, result) );
	    break;

	case iio_space_id:
	    if (offs > 0x7f)
		return iio_error("IPID address out of range");
	    result = (void *)(reg->a16addr + 0x100 * slot + 0x80 + offs);
	    iio_eret(
		iio_operate_addr(reg->bus, iio_space_mem16 | size, &result) 
	    );
	    iio_eret( iio_data_set_addr(opnode, slot, result) );
	    break;

	case iio_space_mem:
	    if (! reg->memory)
		return iio_error("IPMEM addressing disabled");
	    if (offs > 0x7fff)
		return iio_error("IPMEM address out of range");
	    result = (void *)(reg->a24addr + 0x10000 * slot + offs);
	    iio_eret(
		iio_operate_addr(reg->bus, iio_space_mem24 | size, &result) 
	    );
	    iio_eret( iio_data_set_addr(opnode, slot, result) );
	    break;

	default:
	    return iio_error("Space code not supported by channel");
	}
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_vipc610_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * The VIPC610 carrier has itself has no state. We also don't probe for 
     * its existance, but leave it to the individual IP module drivers, which
     * should always probe for their IDPROM and check their identity code
     */
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_vipc610_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    reg->memory = iio_bool_false;
    reg->a16addr = BASE;

    /* decode the arguments */
    iio_eret( iio_arg(argv, "address", iio_arg_uint16, &reg->a16addr) );
    iio_eret( iio_arg(argv, "memory", iio_arg_bool, &reg->memory) );
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &reg->bus) );

    /* this board always uses D16 (I think) */
    if (!reg->bus)
	iio_eret( iio_open("vme.0", 0, &reg->bus) );

    /* do one mapping for the IO space */
    iio_eret( iio_map(reg->bus, iio_space_mem16, reg->a16addr, A16SIZE));

    /* do another mapping for the memory space, if selected */
    reg->a24addr = ((uint32_t)reg->a16addr << 8);
    if (reg->memory)
	iio_eret( iio_map(reg->bus, iio_space_mem24, reg->a24addr, A24SIZE) );

    /* register the four IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_ip, 0, 4,
	    iio_vipc610_ip, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_vipc610(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"vipc610",
	"GreenSpring VIPC610 IP Carrier",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_vipc610_install,
	iio_vipc610_init
    );
}
