/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: tews850.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * tews850.c.c -- driver for TEWS DatenTechnik TIP 850
 *
 * Jonathon Ralston, December 1997
 * CSIRO Division of Exploration and Mining
 *
 * The TEWS TIP 850 provides eight differential or sixteen
 * single-ended 12-bit ADCs, as well as four independent 12-bit DACs.
 * Minimum single channel throughput is 100 kHz.
 *
 * Input ranges are +/-10, +/-5, +/-2.5, or +/-1.25 V (software
 * selectable), and the output range is +/-10V.
 *
 * Module options:
 *
 *	-slot <channel>			IP channel
 *	-gain <gain>			1, 2, 4, or 8.
 *
 * The ADC auto settle flag is polled to ascertain conversion status.
 *
 * The ADC/DAC correction factors stored in the IDPROM are not used.  
 */

#include "../internal.h"


/* 
 * TEWS TIP 850-11 header file 
 */

#if 0
#define ADC_CHAN    8          /* differential ADC configuration */
#else
#define ADC_CHAN   16          /* single-ended ADC configuration */
#endif

#define DAC_CHAN    4          /* DAC channels */


/* Registers */

#define ADCCSR 0x00         /* ADC control and status (word) */
#define ADCCON 0x02         /* ADC convert (word) */
#define ADCDAT 0x04         /* ADC data (word) */
#define DACDA1 0x06         /* DAC 1 data (word) */
#define DACDA2 0x08         /* DAC 2 data (word) */
#define DACDA3 0x0a         /* DAC 3 data (word) */
#define DACDA4 0x0c         /* DAC 4 data (word) */
#define DACLOA 0x0e         /* DAC load  (word) */
#define INTVEC 0x41         /* Interrupt vector register (byte) */


/* 
 * ADC options (write only)
 *
 * CS0-CS7 are valid for both differential and single-ended modes
 * CS8-CS15 are only valid for single-ended mode
 *
 */

#define CS0   0x0000
#define CS1   0x0001
#define CS2   0x0002
#define CS3   0x0003
#define CS4   0x0004
#define CS5   0x0005
#define CS6   0x0006
#define CS7   0x0007

#define CS8   0x0008
#define CS9   0x0009
#define CS10  0x000a
#define CS11  0x000b
#define CS12  0x000c
#define CS13  0x000d
#define CS14  0x000e
#define CS15  0x000f

/* 
 * ADC mode (write only) 
 *
 * DIFFER  differential mode (channels 0-7 only)
 * SINGLE  single endede mode (channels 0-15)
 *
 */

#define DIFFER  0x0010                /* bit 4 */
#define SINGLE  0x0000

/* 
 * ADC gain control selection  (write only)
 *
 * GAIN1  +/- 10 V
 * GAIN2  +/-  5 V
 * GAIN3  +/-  2.5 V 
 * GAIN4  +/-  1.25 V
 *
 */

#define GAIN1  0x0000                  /* bits 5 & 6 */
#define GAIN2  0x0020               
#define GAIN4  0x0040
#define GAIN8  0x0060

/*
 * ADC Automatic settling time control (write only)
 * 
 * Data conversion is automatically delayed in hardware until the
 * settling time has expired. The settling time is gain dependent:
 *  
 * GAIN1 : 52 microsecond delay 
 * GAIN2 : 44 microsecond delay
 * GAIN4 : 44 microsecond delay
 * GAIN8 : 44 microsecond delay
 *
 */

#define ASETTLE 0x0080               /* bit 7 true */
#define POLL    0x0000 
/*
 * ADC Interrupt enable (write only)
 *
 */

#define INT_ENABLE  0x0100           /* bit 8 true */
#define INT_DISABLE 0x0000


/*
 * ADC Status (read only)
 *
 * SETTLE_BUSY is '1' if the AD conversion is settling, '0' on completion.
 * ADC_BUSY is '1' if the ADC is busy, '0' if not busy.
 *
 */

#define SETTLE_BUSY 0x4000               /* bit 14 */
#define ADC_BUSY    0x8000               /* bit 15 */


/* prototypes */

IIO_STATUS iio_tews850( void );
HIDDEN IIO_STATUS iio_tews850_init(IIO_MREG *reg, IIO_MSTATE *state);
HIDDEN IIO_STATUS iio_tews850_install(IIO_MODULE *module, char *argv[]);

HIDDEN IIO_STATUS iio_tews850_dac( IIO_MSTATE *state, IIO_MREG *reg, 
  IIO_OPNODE *opnode,  IIO_OP op, unsigned first, unsigned number );

HIDDEN IIO_STATUS iio_tews850_adc( IIO_MSTATE *state, IIO_MREG *reg, 
  IIO_OPNODE *opnode,  IIO_OP op, unsigned first, unsigned number );




/*
 * Register pointer structure 
 */

struct IIO_MREG {
  IIO slot;			/* the IP slot we are in */
  int gain;                     /* Gain selection */        

  volatile uint16_t             
    *adc_csr,			/* ADC control and status */
    *adc_con,			/* ADC conversion */ 
    *adc_dat,			/* ADC data  */
    *dac_da[DAC_CHAN],		/* four DAC data */
    *dac_load;			/* DAC load */
  
  volatile uint8_t
    *int_vec;                   /* interrupt vector register */

};

/*
 * TEWS 850 state data structure 
 */

struct IIO_MSTATE {
  int      adc_chan;		        /* last ADC channel (one multiplexed chan) */
  int      dac_chan[DAC_CHAN];		/* last DAC channel (separate chans) */
  volatile int16_t dac[DAC_CHAN];	/* data register readback */  
};




/*
 * DAC operation function
 *
 * Write to the DACs as specified.  The TEWS 850 DAC logic is
 * double buffered, and so an explicit DAC load register is used
 * to update outputs.
 */


HIDDEN IIO_STATUS iio_tews850_dac( IIO_MSTATE *state, IIO_MREG *reg, 
  IIO_OPNODE *opnode,  IIO_OP op, unsigned first, unsigned number ) 
{
	uint8_t chan;
	uint16_t val;

	for( chan = first; chan < first + number; ++chan ) {
		switch (op) {
		case iio_op_write:
			/* 
			 * Since the DAC is double buffered, the DAC values 
			 * are first loaded and then converted on writing to 
			 * the DAC load register.  
			 */

			/*
			 * Get DAC channel <chan> value ...
			 */
			val = iio_data_get( opnode, chan );
   	 
			/*
			 * Load DAC register and update DAC channel state cache 
			 */
			(*reg->dac_da[chan]) = state->dac_chan[chan] = (uint16_t)val;
			state->dac[chan] = (int16_t)val;
  
			/* 
			 * Write to DAC load register to initiate DAC data 
			 * conversions.
			 */
			*reg->dac_load = 0;  

			break;
		case iio_op_readback:
			iio_eret(iio_data_set(opnode, chan, (int)state->dac[chan]));
			break;
		default:
			return iio_error("Operation code not supported by channel");
			break;
		}
 	}

	return iio_status_ok;
}

/* ADC operation function
 *
 * Read the specified ADC channels using polling.
 *
 * Although the TEWS TIP850 IP-module has an automatic settling time
 * control register, this is really only useful for a driver which can
 * service interrupts.  Since the IIO library does not support
 * interrupts, only polling is available.  (On the positive side, the
 * ADC chip is quite fast, and therefore little time is wasted in the
 * poll...)
 */

HIDDEN IIO_STATUS iio_tews850_adc( IIO_MSTATE *state, IIO_MREG *reg, 
  IIO_OPNODE *opnode,  IIO_OP op, unsigned first, unsigned number ) 
{
  uint8_t chan;
  int16_t val;
  uint16_t csr;
  
  if (op != iio_op_read)
    return iio_error("Operation code not supported by channel");
  
  /* 
   * Read all specified ADC channels.
   */
  
  for( chan = first; chan < first + number; ++chan ) {
    

    /* Small optimisation: Check to see if the current ADC channel is
     * the same as the last time.  If so, do not update the
     * control/status register.  
     */
    
    csr = INT_DISABLE | ASETTLE | reg->gain | SINGLE | chan; 
    
    *reg->adc_csr = (uint16_t)csr;
    
    if( chan != state->adc_chan ) {
      
      state->adc_chan = chan;            /* update adc channel state cache */
      
      /* 
       * Set ADC control/status register.  See Section 5 of
       * TEWS manual and tews850.h for details.
       */
      
      csr = INT_DISABLE | ASETTLE | reg->gain | SINGLE | chan;
      
      *reg->adc_csr = (uint16_t)csr;
    }

    /*
     * Initiate ADC conversion on channel <chan> immediately.
     *
     * If the automatic settling time control bit is set in the ADC
     * control and status register, then the bus is locked until the
     * conversion is complete.  Alternatively, polling mode is
     * enabled.  Bits 14 and 15 of the ADC CSR register can be read to
     * determine the card/conversion status.
     * 
     * NB: Interrupts cannot be used with the iio library.
     */
    
    *reg->adc_con = (uint16_t)0;          /* initiate conversion */
    
    /* poll status register until ready ... */
    
    while( (*reg->adc_csr) & (SETTLE_BUSY | ADC_BUSY) )
      ;
    
    val = *reg->adc_dat;    /* read the ADC value */
    
    /*
     * Set result
     */
    
    iio_eret( iio_data_set( opnode, chan, (int)val ) );
  }
  
  return iio_status_ok;

}




/* 
 * TEWS 850 initialisation function
 *
 * Initialise the module using the configuration already loaded in the
 * state and register structures. There should be no need to probe, as
 * the IP carrier driver has already done it. Just initialise the
 * registers to match the state (or vice-versa).
 *
 * NB: Hardware module probing (using iio_probe) is not required for
 * IP-modules housed in IP carriers.
 *
 */

HIDDEN IIO_STATUS iio_tews850_init(IIO_MREG *reg, IIO_MSTATE *state) 
{

  uint8_t chan;
  
  /* 
   * Verify the module via the manufacturer ID and model number from
   * the IP module PROM.
   *
   * For the TEWS TIP 850-11, these should be:
   * 
   * Manufacturer ID = 0xb3
   * Model number = 0x09
   * */
  
  iio_eret( iio_ipinfo_ident(reg->slot, 0xb3, 0x11 ) );

  /*
   * set the initial ADC and DAC channels to "nonsense" values 
   */
  
  state->adc_chan = -1;

  for( chan=0; chan < ADC_CHAN; chan++ )
    state->dac_chan[chan] = -1;

  /*
   * Reset all DAC outputs to zero.
   */
  
  for( chan = 0; chan < DAC_CHAN; chan++ ) {
    
    /*
     * Load DAC register 
     */
    
    (*reg->dac_da[chan]) = 0; 
    state->dac[chan] = 0;
  }

    
  /* 
   * Write to DAC load register to initiate DAC data conversions.
   */
  
  *reg->dac_load = 0;    
  
  return iio_status_ok;
}




/*
 * TEWS 850 Installation function
 */


HIDDEN IIO_STATUS iio_tews850_install(IIO_MODULE *module, char *argv[]) {
  
  /*
   * Decode the module driver arguments, resolve the VME base address to
   * logical addresses, build the register structure containing pointers 
   * to the device, and register the channels it provides.
   */
  
  IIO_CHNODE *chnode;
  IIO_MREG *reg;
  int chan;
  double gain = 1.0;
    
  /* get a zeroed register structure */
  iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
  
  /* get a state structure */
  iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );
  
  /*
   * Get module specific configuration details 
   * 
   * NB: Slot specification is mandatory.
   *
   */
  
  iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );

  /* /etc/iio.conf UNIX command-style options */

  iio_eret( iio_arg(argv, "gain", iio_arg_double, &gain) );
  
  if (!reg->slot)
    return iio_error("No slot specified");
  
  if( gain == 1 ) 
    reg->gain = GAIN1;
  else if( gain == 2 ) 
    reg->gain = GAIN2;
  else if( gain == 4 ) 
    reg->gain = GAIN4;
  else if( gain == 8 ) 
    reg->gain = GAIN8;
  else
    return iio_error( "Invalid gain option: Must be 1, 2, 4, or 8" );

  /* 
   * Resolve local register addresses. See register pointer structure
   * for details...
   *
   */
  
  iio_eret( 
	   iio_resolve_list(
			    reg->slot, iio_space_io,
			    iio_size_16, 0x00, &reg->adc_csr,
			    iio_size_16, 0x02, &reg->adc_con,
			    iio_size_16, 0x04, &reg->adc_dat,
			    iio_size_16, 0x06, &reg->dac_da[0],
			    iio_size_16, 0x08, &reg->dac_da[1],
			    iio_size_16, 0x0A, &reg->dac_da[2],
			    iio_size_16, 0x0C, &reg->dac_da[3],
			    iio_size_16, 0x0E, &reg->dac_load,
			    iio_size_8,  0x41, &reg->int_vec,
			    0
			    )
	   );
  
  /* 
   *  Register the ADC and DAC channels with the channel list 
   */
  
  
  iio_eret(
	   iio_chnode(
		      module,
		      iio_chtype_adc, 12, ADC_CHAN, 
		      iio_tews850_adc, &chnode
		      )
	   );


  
  /* set the gain conversions for all channels */
  for (chan = 0; chan < ADC_CHAN; ++chan)
    iio_eret(
	     iio_chnode_linear(
			       chnode, chan,
			       20.0/4096.0,         /* *gain   */
			       0.0,
			       "V"
			       )
	     );
  
  
  iio_eret(
	   iio_chnode(
		      module,
		      iio_chtype_dac, 12, DAC_CHAN, 
		      iio_tews850_dac, &chnode
		      )
	   );
  
  
  /* set the gain conversions for all channels */
  for (chan = 0; chan < DAC_CHAN; ++chan)
    iio_eret(
	     iio_chnode_linear(
			       chnode, chan,
			       20.0/4096.0,
			       0.0,
			       "V"
			       )
	     );
  
  return iio_status_ok;
}



/*
 * TEWS 850 Identification function
 *
 * Call iio_minfo to register this module with IIO
 *
 */

IIO_STATUS iio_tews850( void ) {
  return iio_minfo(
		   "tews850",
		   "TEWS DatenTechnik TIP 850",
		   "$Revision: 2222 $",
		   iio_multi_yes,
		   iio_tews850_install,
		   iio_tews850_init
		   );
}


