/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: pcpp.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * pcpp - driver for the ISA PC parallel port (including IEEE 1284)
 *
 * The generic PC parallel port provides 1 8-bit digital output
 * port, one 5-bit digital input port and one 5-bit digital i/o port.
 * In the more advanced EPP and ECP modes, it provides a bi-directional
 * addr/data port or a command/data port, but compatible hardware is needed
 *
 *	-bus <channel>			ISA bus
 *	-address <addr>			Base address of the IO port
 *      -mode <mode>                    one of spp, epp etc
 *      -irq <num>                      currently not used
 */

#include "../internal.h"

    /* default base address */
#define BASE		0x378

    /* SPP status register bits */
#define SPP_STATUS_BUSY		0x80
#define SPP_STATUS_ACK		0x40
#define SPP_STATUS_PE		0x20
#define SPP_STATUS_SELIN	0x10
#define SPP_STATUS_ERR		0x08
#define SPP_STATUS_IRQ		0x04
#define SPP_STATUS_EPP_TIMEOUT	0x01

#define SPP_STATUS_ext_mask	0xf8
#define SPP_STATUS_ext_roll	3

    /* SPP control register bits */
#define SPP_CONTROL_INPUT	0x20	/* allegedly 0x40 on some hardware! */
#define SPP_CONTROL_INTEN	0x10
#define SPP_CONTROL_SEL		0x08
#define SPP_CONTROL_INIT	0x04
#define SPP_CONTROL_AFD		0x02
#define SPP_CONTROL_STR		0x01

#define SPP_CONTROL_ext		0x0f

    /* ECP ECR register bits */
#define ECP_ECR_MODE_SPP	0x00
#define ECP_ECR_MODE_SPPR	0x20
#define ECP_ECR_MODE_SPPFIFO	0x40
#define ECP_ECR_MODE_ECP	0x60
#define ECP_ECR_MODE_EPP	0x80
#define ECP_ECR_MODE_FIFOTEST	0xb0
#define ECP_ECR_MODE_CONFIG	0xe0
#define ECP_ECR_INTEN		0x10
#define ECP_ECR_DMAEN		0x08
#define ECP_ECR_SERVICE		0x04
#define ECP_ECR_FIFOFULL	0x02
#define ECP_ECR_FIFOEMPTY	0x01


    /* port modes */
typedef enum {
    pcpp_spp, pcpp_sppi, pcpp_epp, pcpp_eppa
} IIO_PCPP_MODE;

    /* names of port modes, must match above, with NULL */
HIDDEN char *iio_pcpp_modes[] = {
    "spp", "sppi", "epp", "eppa", NULL
};

    /* register data structure */
struct IIO_MREG {
    IIO_PCPP_MODE mode;		/* port mode */

    /* "standard" parallel port registers */
    volatile uint8_t
        *data,			/* SPP data register */
        *status,		/* SPP status register */
        *control;		/* SPP control register */

    /* enhanced parallel port registers */
    volatile uint8_t
        *epp_addr,		/* EPP address register */
        *epp_data;		/* EPP data register */

    /* extended capability port registers */
    volatile uint8_t
        *ecp_data,		/* ECP data register */
        *ecp_config_a,		/* ECP config register A */
        *ecp_config_b,		/* ECP config register B */
        *ecp_ecr;		/* ECP extended capability register */
};


    /* state data structure */
struct IIO_MSTATE {
    uint8_t
	spp_control,		/* control port shadow */
	epp_rdio[2],		/* addr/data port shadow, EPP mode */
	eppa_do[256];		/* data port shadows, EPPA mode */
};


HIDDEN IIO_STATUS iio_pcpp_spp_oco(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * SPP and SPPI MODE
     *
     * Operation function for the four do bits accessed through the
     * control port of the standard parallel port. These are used as
     * plain bits, not for handshaking. Some outputs are inverted.
     *
     * We have to preserve the upper bits, especially the bidirectional bit.
     * We could read the port back, but it's much quicker to keep a copy
     * in the state structure.
     *
     * Some references claim that these bits are really OCIO
     */
    switch (op) {
    case iio_op_readback:
	iio_data_set(
	    opnode, first,
	    SPP_CONTROL_ext & (state->spp_control ^ SPP_CONTROL_INIT)
	);
	break;
    
    case iio_op_write:
	state->spp_control = (
		SPP_CONTROL_ext &
		(iio_data_get(opnode, first) ^ SPP_CONTROL_INIT)
	    ) | (~SPP_CONTROL_ext & state->spp_control);

	iio_port_set8(reg->control, state->spp_control);
	break;
    
    case iio_op_clear:
	state->spp_control =
	    (~SPP_CONTROL_ext & state->spp_control) | SPP_CONTROL_INIT;
	iio_port_set8(reg->control, state->spp_control);
	break;
    
    default:
	return iio_error("Operation code not supported by channel");
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_spp_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * SPP and SPPI MODE
     * 
     * Operation function for the five di bits accessed through the
     * status port of the standard parallel port. These are used as
     * plain bits, not for handshaking.
     *
     * The BUSY bit is hardware inverted, so we invert it to bring
     * it into line with the others
     */
    if (op == iio_op_read) {
	iio_data_set(
	    opnode, first,
	    (SPP_STATUS_BUSY ^ iio_port_get8(reg->status))
		>> SPP_STATUS_ext_roll
	);
    } else
	return iio_error("Operation code not supported by channel");

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_spp(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * SPP MODE
     * This is the operation function used in the unidirectional SPP mode.
     * The 8 bits drive a totem-pole DO buffer
     */
    switch (op) {
    case iio_op_readback:
	iio_data_set(opnode, first, iio_port_get8(reg->data));
	break;
    
    case iio_op_write:
	/* set the data first */
	iio_port_set8(reg->data, iio_data_get(opnode, first));
	break;
    
    case iio_op_clear:
	iio_port_set8(reg->data, 0);
	break;
    
    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_sppi(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * SPPI MODE
     * Operation function for the eight data di bits. The output buffer
     * was tri-started by the initialisation function
     */
    if (op == iio_op_read) {
	iio_data_set(opnode, first, iio_port_get8(reg->data));
    } else
	return iio_error("Operation code not supported by channel");

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_epp(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * EPP MODE 
     * This provides two 8-bit drio ports, address (0) and data (1). The
     * two ports can be differentiated by hardware the uses the separate
     * strobe lines used in this mode.
     *
     * FIXME check the timeout bit
     */
    int seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_readback:
	    iio_data_set(opnode, seqno, state->epp_rdio[seqno]);
	    break;

	case iio_op_read:
	    iio_data_set(
		opnode, seqno,
		iio_port_get8((seqno ? reg->epp_addr : reg->epp_data))
	    );
	    break;
	
	case iio_op_write:
	    iio_port_set8(
		(seqno ? reg->epp_addr : reg->epp_data), 
		state->epp_rdio[seqno] = iio_data_get(opnode, seqno)
	    );
	    break;
	
	case iio_op_clear:
	    iio_port_set8(
		(seqno ? reg->epp_addr : reg->epp_data),
		state->epp_rdio[seqno] = 0
	    );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_eppa(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * EPPA MODE
     * This uses the address port as an address to select one of 256
     * data registers. As with the other EPP mode, it is assumed there
     * is hardware of a certain kind connected to the port. (You can use
     * the EPP mode to do the same thing, but this mode is a little quicker,
     * and you don't need an external mutex)
     */
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Perform once-only initialisation or port, which depends
     * on the mode it was installed with
     */
    uint8_t cr, ecr;

#if 0
    iio_log("port 0x%03x:\n", SPP_DATA);
    iio_log("    dr = 0x%02x\n", iio_port_get8(reg->data));
    iio_log("    sr = 0x%02x\n", iio_port_get8(reg->status));
    iio_log("    cr = 0x%02x\n", iio_port_get8(reg->control));
    iio_log "    ecr = 0x%02x\n", iio_port_get8(reg->ecp_ecr));
#endif

    /*
     * First, check/set the mode. We can't really do much unless the 
     * port is ECP capable (i.e., has an ECR), so we just accept the
     * mode as specified by the user.
     *
     * We can test for an ECR by reading it, and comparing with the CR.
     * If it is the same, it's old 8-bit hardware (the address
     * wraps around 0x400). If we get 0xff, it is either 16-bit non-ECR
     * hardware, or potentially ECR-capable hardware not in ECR capable
     * mode(!). If we get something else, it's probably the ECR, so it's
     * ECR capable hardware. BUT, the ECR gives the actual hardware mode.
     * 
     * It would appear ECR hardware is initially in SPP mode, so we have
     * to change it to whatever we want. We don't actually support
     * ECP mode as such anyway
     */

    /* read CR and ECR */
    cr = iio_port_get8(reg->control);
    ecr = iio_port_get8(reg->ecp_ecr);

    /* look for behaviour suggesting ECP capability */
    if ((cr != ecr) && (ecr != 0xff)) {

	/* assume ECP capable hardware */
	switch (reg->mode) {
	case pcpp_spp:
	    iio_port_set8(reg->ecp_ecr, ECP_ECR_MODE_SPP);
	    break;

	case pcpp_sppi:
	    iio_port_set8(reg->ecp_ecr, ECP_ECR_MODE_SPPR);
	    break;

	case pcpp_epp:
	case pcpp_eppa:
	    iio_port_set8(reg->ecp_ecr, ECP_ECR_MODE_EPP);
	    break;
	}
    } 


    /*
     * Clear control outputs -- they will actually float high.
     * In SPPI mode, also switch off data buffers
     */
    state->spp_control =
	SPP_CONTROL_INIT |
	((reg->mode == pcpp_sppi) ? SPP_CONTROL_INPUT : 0);
    iio_port_set8(reg->control, state->spp_control);


    /* clear data outputs and shadows */
    switch (reg->mode) {
    case pcpp_spp:
    case pcpp_sppi:
	iio_port_set8(reg->data, 0);
	break;

    case pcpp_epp:
	state->epp_rdio[0] = 0;
	state->epp_rdio[1] = 0;
	iio_port_set8(reg->data, 0);
	iio_port_set8(reg->epp_data, 0);
	iio_port_set8(reg->epp_addr, 0);
	break;

    case pcpp_eppa:
	/* FIXME should clear all 128 do registers */
	break;
    }

#if 0
    iio_log("    ecr = 0x%02x after init\n", iio_port_get8(reg->ecp_ecr));
#endif

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcpp_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments and register the channels it
     * provides. This depends on the mode (SPP, EPP etc) 
     */
    IIO chan = NULL;
    IIO_MREG *reg;
    IIO_CHNODE *chnode;
    uint8_t irq = 7;
    uint16_t base = BASE;
    char *mode = "spp";

    /* allocate register and state structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* get the arguments */
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &chan) );
    iio_eret( iio_arg(argv, "address", iio_arg_uint16, &base) );
    iio_eret( iio_arg(argv, "mode", iio_arg_string, &mode) );
    iio_eret( iio_arg(argv, "irq", iio_arg_uint8, &irq) );

    /* decode mode argument */
    if (iio_string_lookup(mode, iio_pcpp_modes, (int *)&(reg->mode)))
	return iio_error("Bad port mode");

    /* check given address is valid */
    switch (base) {
    case 0x3bc:
    case 0x378:
    case 0x278:
	/* OK for any mode */
	break;

    default:
	return iio_error("Bad port address");
    }

    /* this module always uses the ISA Bus */
    if (!chan)
	iio_eret( iio_open("isa.0", 0, &chan) );

    /* map registers -- there are two discontiguous areas */
    iio_eret( iio_map(chan, iio_space_port, base, 5) );
    iio_eret( iio_map(chan, iio_space_port, base + 0x400, 3) );

    /* resolve registers (ecp_data and ecp_config_a are the same address) */
    iio_eret(
	iio_resolve_list(
	    chan, iio_space_port,
	    iio_size_8, base + 0x000, &reg->data,
	    iio_size_8, base + 0x001, &reg->status,
	    iio_size_8, base + 0x002, &reg->control,
	    iio_size_8, base + 0x003, &reg->epp_addr,
	    iio_size_8, base + 0x004, &reg->epp_data,
	    iio_size_8, base + 0x400, &reg->ecp_data,
	    iio_size_8, base + 0x400, &reg->ecp_config_a,
	    iio_size_8, base + 0x401, &reg->ecp_config_b,
	    iio_size_8, base + 0x402, &reg->ecp_ecr,
	    0
	)
    );
    iio_close(chan);

    /* register the IO resources, which depend on the mode */
    switch (reg->mode) {
    case pcpp_spp:
    case pcpp_sppi:
	/*
	 * In SPP and SPPI mode we use the status bits to give
	 * 5 di channels, and the control bits to give 4 oco channels.
	 * For SPP the 8 data bits are do but SPPI they are di
	 */

	/* five status bits */
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_di, 5, 1,
		iio_pcpp_spp_di, &chnode
	    )
	);

	/* four control bits */
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_oco, 4, 1,
		iio_pcpp_spp_oco, &chnode
	    )
	);

	/* eight data bits */
	switch (reg->mode) {
	case pcpp_spp:
	    iio_eret(
		iio_chnode(
		    module,
		    iio_chtype_do, 8, 1,
		    iio_pcpp_spp, &chnode
		)
	    );
	    break;

	case pcpp_sppi:
	    iio_eret(
		iio_chnode(
		    module,
		    iio_chtype_di, 8, 1,
		    iio_pcpp_sppi, &chnode
		)
	    );
	    break;

	default:
	    break;
	}
	break;

    case pcpp_epp:
	/*
	 * The port in EPP mode provides two 8-bit rdio ports (address and
	 * data). The data for the two ports comes out on the same data
	 * lines; the difference is in which strobe is used. You need some
	 * extra hardware to separate the data
	 */
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_rdio, 8, 2,
		iio_pcpp_epp, &chnode
	    )
	);
	break;

    case pcpp_eppa:
	/*
	 * The "addressable" EPPA mode assumes the address register is
	 * actually used an an address register to select amongst 256
	 * data in and data out registers. The address is set before
	 * each operation
	 */
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_do, 8, 128,
		iio_pcpp_eppa, &chnode
	    )
	);
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_di, 8, 128,
		iio_pcpp_eppa, &chnode
	    )
	);
	break;
    }
   
    return iio_status_ok;
}


IIO_STATUS iio_pcpp(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"pcpp",
	"PC Parallel Port",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_pcpp_install,
	iio_pcpp_init
    );
}
