/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: pcgp.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * pcgp - driver for the ISA PC game port
 *
 * The generic PC parallel game provides 4 potentiometer inputs
 * two per player) and 4 digital inputs. However, we can't support the
 * potentiomenter inputs without a timing loop
 *
 *	-bus <channel>			ISA bus
 */

#include "../internal.h"

    /* register data structure */
struct IIO_MREG {
    volatile uint8_t *base;		/* the one register */
};


HIDDEN IIO_STATUS iio_pcgp_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * The switches are the top 4 bits in the port
     */
    if (op == iio_op_read) {
	iio_data_set(opnode, first, iio_port_get8(reg->base) >> 4);
    } else
	return iio_error("Operation code not supported by channel");

    return iio_status_ok;
}


#if 0
HIDDEN IIO_STATUS iio_pcgp_adc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * We don't really support the ADCs, which aren't ADCs at all
     */
    unsigned seqno;

    if (op == iio_op_read) {
	for (seqno = first; seqno < first + number; ++seqno) {
	    iio_data_set(
		opnode, first,
		0
	    );
	}
    } else
	return iio_error("Operation code not supported by channel");

    return iio_status_ok;
}
#endif


HIDDEN IIO_STATUS iio_pcgp_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * No initialisation required
     */
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_pcgp_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments and register the channels it
     * provides. 
     */
    IIO chan = NULL;
    IIO_MREG *reg;
    IIO_CHNODE *chnode;

    /* allocate register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* this module always uses the ISA Bus */
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &chan) );
    if (! chan)
	iio_eret( iio_open("isa.0", 0, &chan) );

    /* map and resolve the one fixed register */
    iio_eret( iio_map(chan, iio_space_port, 0x201, 1) );
    iio_eret(
	iio_resolve(
	    chan, iio_space_port, iio_size_8,
	    0x201, (void **)&reg->base
	)
    );
    iio_close(chan);

    /* register the IO resources */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_di, 4, 1,
	    iio_pcgp_di, &chnode
	)
    );
#if 0
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 8, 4,
	    iio_pcgp_adc, &chnode
	)
    );
#endif

    return iio_status_ok;
}


IIO_STATUS iio_pcgp(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"pcgp",
	"PC Game Port",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_pcgp_install,
	iio_pcgp_init
    );
}
