/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: modbus.c 3102 2008-05-18 22:52:04Z roy029 $
 *
 * modbus.c - functions for MODBUS modules
 * Pavan Sikka, September 2003
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#include <errno.h>

#include "../internal.h"

/* RTX_USED requires rtx/defines.h, but that would cause a cyclic dependancy between iio and rtx.
#include <rtx/defines.h>

static char rcsid[] RTX_UNUSED = "$ID$";
*/

#define IIO_MODBUS_TCP_PORT          502
#define IIO_MODBUS_ADU_LEN_MAX       256

#define IIO_MODBUS_ILLEGAL_FUNCTION_EXCEPTION                            0x01
#define IIO_MODBUS_ILLEGAL_DATA_ADDRESS_EXCEPTION                        0x02
#define IIO_MODBUS_ILLEGAL_DATA_VALUE_EXCEPTION                          0x03
#define IIO_MODBUS_SLAVE_DEVICE_FAILURE_EXCEPTION                        0x04
#define IIO_MODBUS_ACKNOWLEDGE_EXCEPTION                                 0x05
#define IIO_MODBUS_SLAVE_DEVICE_BUSY_EXCEPTION                           0x06
#define IIO_MODBUS_MEMORY_PARITY_ERROR_EXCEPTION                         0x08
#define IIO_MODBUS_GATEWAY_PATH_UNAVAILABLE_EXCEPTION                    0x0A
#define IIO_MODBUS_GATEWAY_TARGET_DEVICE_FAILED_TO_RESPOND_EXCEPTION     0x0B

IIO_STATUS
iio_modbus_connect (
		    char * hostname,
		    int * sockp
		    )
{
    struct sockaddr_in saddr;
    struct sockaddr_in daddr;
    struct hostent * host;
    char errstr[256];

    if ((* sockp = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
        strcpy (errstr, "iio_modbus_connect[socket()]: ");
	strcat (errstr, strerror (errno));
        return (iio_error (errstr));
    }
    if ((host = gethostbyname (hostname)) == NULL) {
        strcpy (errstr, "iio_modbus_connect[gethostbyname()]: ");
	strcat (errstr, strerror (errno));
        return (iio_error (errstr));
    }
    memset (&saddr, 0, sizeof (saddr));
    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = htonl (INADDR_ANY);
    saddr.sin_port = 0;
    if (bind (* sockp, (struct sockaddr *) &saddr, sizeof (saddr))
	    == -1) {
        strcpy (errstr, "iio_modbus_connect[bind()]: ");
	strcat (errstr, strerror (errno));
        return (iio_error (errstr));
    }
    memset (&daddr, 0, sizeof (saddr));
    daddr.sin_family = AF_INET;
    daddr.sin_addr = * (struct in_addr *) host->h_addr;
    daddr.sin_port = htons (IIO_MODBUS_TCP_PORT);
    if (connect (* sockp, (struct sockaddr *) &daddr, sizeof (daddr))
	    == -1) {
        strcpy (errstr, "iio_modbus_connect[connect()]: ");
	strcat (errstr, strerror (errno));
        return (iio_error (errstr));
    }
    return (iio_status_ok);
}

IIO_STATUS
iio_modbus_disconnect (
		       int sock
		       )
{
    shutdown (sock, 2);
    close (sock);
    return (iio_status_ok);
}

/*
mbus read command:

cmd[0 ] = tid upper byte (usually 0)
cmd[1 ] = tid lower byte (usually 0)
cmd[2 ] = protoid upper byte (always 0)
cmd[3 ] = protoid lower byte (always 0)
cmd[4 ] = len upper byte (always 0)
cmd[5 ] = len lower byte (number of bytes following)
cmd[6 ] = unit id
cmd[7 ] = function code (3)
cmd[8 ] = reg no upper byte
cmd[9 ] = reg no lower byte
cmd[10] = reg count (upper byte)
cmd[11] = reg count (lower byte)

Note: registers are 16-bit wide
*/

IIO_STATUS
iio_modbus_read (
		 int sock,
		 int unitid,
		 int modFunc,
		 int regnum,
		 int count,
		 short reg[]
		 )
{
	int n, i = 0;
	int pktLen = IIO_MODBUS_ADU_LEN_MAX;
	unsigned char buf[512];
	char errstr[256];
	char msgbyte[256];

	/* zero the leading part of the header */
	for (i=0; i<5; i++)
		buf[i] = 0;
	buf[5] = 6;
	buf[6] = unitid & 0xFF;
	buf[7] = modFunc;
	buf[8] = (regnum >> 8) & 0xFF;
	buf[9] = regnum & 0xFF;
	buf[10] = (count >> 8) & 0xFF;
	buf[11] = count & 0xFF;
	if (write (sock, buf, 12) == -1) {
		strcpy (errstr, "iio_modbus_read[write()]: ");
		strcat (errstr, strerror (errno));
		return (iio_error (errstr));
	}
	for (i = 0; i < pktLen; ) {
		if ((n = read (sock, &(buf[i]), pktLen)) == -1) {
			strcpy (errstr, "iio_modbus_read[read()]: ");
			strcat (errstr, strerror (errno));
			return (iio_error (errstr));
		}
		i += n;
		if (i >= 6)
			pktLen = buf[5] + 6;
	}

	if (buf[7] != modFunc) {
		if ((buf[7] & 0x7F) == modFunc) {
			/* MODBUS error */
			sprintf (errstr, "iio_modbus_read: modbus error code = %d",
					buf[8]);
			return (iio_error (errstr));
		} else {
			strcat (errstr, "iio_modbus_read: unknown message [");
			for (i=0; i<pktLen; i++) {
				sprintf (msgbyte, "%02X ", (unsigned char) buf[i]);
				strcat (errstr, msgbyte);
			}
			strcat (errstr, "]");
			return (iio_error (errstr));
		}
	}
	/* We have the data message */
	for (i=0; i<count; i++) {
		reg[i] = (buf[9+2*i] << 8) | buf[9+2*i+1];
	}
	return (iio_status_ok);
}

/*
mbus write command:

cmd[0 ] = tid upper byte (usually 0)
cmd[1 ] = tid lower byte (usually 0)
cmd[2 ] = protoid upper byte (always 0)
cmd[3 ] = protoid lower byte (always 0)
cmd[4 ] = len upper byte (always 0)
cmd[5 ] = len lower byte (number of bytes following)
cmd[6 ] = unit id
cmd[7 ] = function code (10)
cmd[8 ] = reg no upper byte
cmd[9 ] = reg no lower byte
cmd[10] = reg count (upper byte)
cmd[11] = reg count (lower byte)
cmd[12] = byte count (2 * reg_count)
...     = register values

Note: registers are 16-bit wide
*/

int
iio_modbus_write (
		  int sock,
		  int unitid,
		  int regnum,
		  int count,
		  short reg[]
		  )
{
    int n, i = 0;
    int pktLen = IIO_MODBUS_ADU_LEN_MAX;
    unsigned char buf[512];
    char errstr[256];
    char msgbyte[256];

    /* zero the leading part of the header */
    for (i=0; i<5; i++)
        buf[i] = 0;
    buf[5] = 6 + count * 2 + 1;
    buf[6] = unitid & 0xFF;
    buf[7] = iio_modbus_write_fn;
    buf[8] = (regnum >> 8) & 0xFF;
    buf[9] = regnum & 0xFF;
    buf[10] = (count >> 8) & 0xFF;
    buf[11] = count & 0xFF;
    buf[12] = (2 * count) & 0xFF;

    for (i=0; i<count; i++) {
        buf[13+2*i] = (reg[i] >> 8) & 0xFF;
        buf[13+2*i+1] = reg[i] & 0xFF;
    }
	
    if (write (sock, buf, 6 + 6 + 2 * count + 1) == -1) {
        strcpy (errstr, "iio_modbus_write[write()]: ");
	strcat (errstr, strerror (errno));
	return (iio_error (errstr));
    }

	for (i = 0; i < pktLen; ) {
		if ((n = read (sock, &(buf[i]), pktLen)) == -1) {
			strcpy (errstr, "iio_modbus_write[read()]: ");
			strcat (errstr, strerror (errno));
			return (iio_error (errstr));
		}
		i += n;
		if (i >= 6)
			pktLen = buf[5] + 6;
	}
	if (buf[7] != iio_modbus_write_fn) {
		if ((buf[7] & 0x7F) == iio_modbus_write_fn) {
			/* MODBUS error */
			sprintf (errstr, "iio_modbus_write: modbus error code = %d",
					buf[8]);
			return (iio_error (errstr));
	} else {
	    strcat (errstr, "iio_modbus_write: unknown message [");
	    for (i=0; i<pktLen; i++) {
	         sprintf (msgbyte, "%02X ", (unsigned char) buf[i]);
		 strcat (errstr, msgbyte);
	    }
	    strcat (errstr, "]");
	    return (iio_error (errstr));
	}
    }
    return (iio_status_ok);
}

IIO_STATUS
iio_modbus_print_packet (
			 char * buf,
			 int n
			 )
{
    int i;

    for (i=0; i<n; i++)
        fprintf (stderr, "%02X ", (unsigned char) buf[i]);
    fprintf (stderr, "\n");
    return (iio_status_ok);
}

#define MOMENTUM_DATA_INPUT_REG                         0x0001
#define MOMENTUM_DATA_OUTPUT_REG                        0x0001

#define MOMENTUM_CONFIG_MODULE_TIMEOUT_REG              0xF001
#define MOMENTUM_CONFIG_MODULE_TIMEOUT_REG_LEN          1
#define MOMENTUM_CONFIG_MODULE_OWNERSHIP_REG            0xF401
#define MOMENTUM_CONFIG_MODULE_OWNERSHIP_REG_LEN        6
#define MOMENTUM_CONFIG_IP_ADDR_SAVED_REG               0xF411
#define MOMENTUM_CONFIG_IP_ADDR_SAVED_REG_READ_LEN      2
#define MOMENTUM_CONFIG_IP_ADDR_SAVED_REG_WRITE_LEN     1

#define MOMENTUM_MODULE_STATUS_REG                      0xF801
#define MOMENTUM_MODULE_STATUS_REG_LEN                  13
#define MOMENTUM_MODULE_ASCII_HEADER_REG                0xFC01

#define MOMENTUM_STATUS_HEALTHY                         0x8000

IIO_STATUS
iio_modbus_momentum_open (
			  char * hostname,
			  int * sockp,
			  IIO_MODBUS_MOMENTUMINFO * info
			  )
{
    int i;
    char hexCh;
    char errstr[256];
    short statusReg[256];

    if (iio_modbus_connect (hostname, sockp) != iio_status_ok) {
        strcpy (errstr, "iio_modbus_momentum_open[iio_modbus_connect()]: ");
	strcat (errstr, iio_emessage_get ());
	return (iio_error (errstr));
    }

    /* Read the module status block */
    if (iio_modbus_read (* sockp, 0, iio_modbus_read_fn, MOMENTUM_MODULE_STATUS_REG - 1,
		   MOMENTUM_MODULE_STATUS_REG_LEN, statusReg)
	    != iio_status_ok) {
        strcpy (errstr, "iio_modbus_momentum_open[iio_modbus_read()]: ");
	strcat (errstr, iio_emessage_get ());
	return (iio_error (errstr));
    }
    /* We have the status message */
    info->len = statusReg[0];
    info->inputLen = statusReg[1];
    info->outputLen = statusReg[2];
    info->moduleId = statusReg[3];
    hexCh = (statusReg[4] >> 8) & 0x0F;
    if (hexCh > 9)
        info->rev[0] = 'A' + hexCh - 10;
    else
        info->rev[0] = '0' + hexCh;
    info->rev[1] = '.';
    hexCh = (statusReg[4] >> 4) & 0x0F;
    if (hexCh > 9)
        info->rev[2] = 'A' + hexCh - 10;
    else
        info->rev[2] = '0' + hexCh;
    hexCh = statusReg[4] & 0x0F;
    if (hexCh > 9)
        info->rev[3] = 'A' + hexCh - 10;
    else
        info->rev[3] = '0' + hexCh;
    info->rev[4] = '\0';
    info->headerLen = statusReg[5];
    sprintf (info->clientIpAddr, "%u.%u.%u.%u",
	     (unsigned int) (statusReg[6] >> 8) & 0xFF,
	     (unsigned int) statusReg[6] & 0xFF,
	     (unsigned int) (statusReg[12] >> 8) & 0xFF,
	     (unsigned int) statusReg[12] & 0xFF);
    info->ownershipResTimeLeft = statusReg[7];
    info->outputsHoldupTimeLeft = statusReg[8];
    info->moduleHealth = statusReg[9];
    info->lastErrorNumber = statusReg[10];
    info->errorCounter = statusReg[11];
    /* Read the module ascii header block */
    if (iio_modbus_read (* sockp, 0, iio_modbus_read_fn, MOMENTUM_MODULE_ASCII_HEADER_REG - 1,
			 info->headerLen, statusReg) != iio_status_ok) {
        strcpy (errstr, "iio_modbus_momentum_open[iio_modbus_read()]: ");
	strcat (errstr, iio_emessage_get ());
	return (iio_error (errstr));
    }
    /* We have the ascii header */
    for (i=0; i<info->headerLen; i++) {
        info->headerBlock[2*i] = (statusReg[i] >> 8) & 0xFF;
        info->headerBlock[2*i+1] = statusReg[i] & 0xFF;
    }
    info->headerBlock[2*(info->headerLen)] = '\0';
    if (info->moduleHealth == MOMENTUM_STATUS_HEALTHY)
        return (iio_status_ok);
    return (iio_error ("iio_modbus_momentum_open: module not healthy"));
}

#define ADVANSYS_STATUS_REG_LEN 1
#define ADVANSYS_STATUS_REG 45391
#define ADVANSYS_ERROR_REG_LEN 1
#define ADVANSYS_ERROR_REG 45358

#define ADVANSYS_MODULE_CONFIG_REG 45359
#define ADVANSYS_MODULE_CONFIG_REG_LEN 2

int crazy_advansys_read_conversion(int address) {
	address &= 0xFF;
	address += 9;
	return address;
}

IIO_STATUS
iio_modbus_advansys_open (
			  char * hostname,
			  int * sockp,
			  IIO_MODBUS_ADVANSYSINFO * info
			  )
{
	int error = 0;
	char errstr[256]="";
	short statusReg;
	if (iio_modbus_connect (hostname, sockp) != iio_status_ok) {
		strcpy (errstr, "iio_modbus_advansys_open[iio_modbus_connect()]: ");
		printf("connect error %s\n",errstr); 
		strcat (errstr, iio_emessage_get ());
		return (iio_error (errstr));
	}

	/* Read the module status block */
	if (iio_modbus_read (* sockp, 0, iio_modbus_input_read_fn, 
				crazy_advansys_read_conversion(ADVANSYS_STATUS_REG),
				ADVANSYS_STATUS_REG_LEN, &statusReg)
			!= iio_status_ok) {
		strcpy (errstr, "iio_modbus_advansys_open[iio_modbus_read()]: ");
		printf("read module status error %s\n",errstr); 
		strcat (errstr, iio_emessage_get ());
		return (iio_error (errstr));
	}

	/* We have the status message */
	if((info->moduleFailure =    statusReg & 0x0100) != 0) {
		sprintf(errstr, "iio_modbus_advansys_open[status] module failure\r\n"); 
		error = 1;
	}
	if ((info->internalFailure =  statusReg& 0x0200) != 0) {
		//sprintf(errstr, "iio_modbus_advansys_open[status]internal failure\r\n"); 
		//fprintf(stderr, "iio_modbus_advansys_open[status]internal failure\r\n"); 
		//error = 1;
	}
	if ((info->externalFailure =  statusReg& 0x0400) != 0){
		sprintf(errstr, "iio_modbus_advansys_open[status]external failure\r\n");
		error = 1;
	}
	if ((info->configProtected =  statusReg& 0x0800) != 0)
		printf("config Protected\r\n");
	if ((info->configInvalid = statusReg& 0x1000) != 0) {
		sprintf(errstr, "iio_modbus_advansys_open[status]config invalid\r\n");
		error = 1;
	}
	if ((info->reflexConfigured = statusReg& 0x2000) != 0)
		printf("reflex configured\r\n");
	if ((info->islandSwapped =    statusReg& 0x4000) != 0)
		printf("island swapped\r\n");
	if ((info-> advansysControl = statusReg& 0x8000) != 0)
		printf("advansys Control\r\n");

	printf("%s\n", errstr);
	/* Read the module error registor */
	if (iio_modbus_read (* sockp, 0, iio_modbus_input_read_fn, 
				crazy_advansys_read_conversion(ADVANSYS_ERROR_REG),
				ADVANSYS_ERROR_REG_LEN, &statusReg) != iio_status_ok) {
		strcpy (errstr, "iio_modbus_advansys_open[iio_modbus_read()]: ");
	printf("%s\n", errstr);
		strcat (errstr, iio_emessage_get ());
		return (iio_error (errstr));
	}

	if (statusReg != 0) {
		sprintf (errstr, "iio_modbus_advansys_open[error reg check]: ERROR SET %02X", statusReg);
		error = 1;
	}	
	if (error) {
		strcat (errstr, iio_emessage_get ());
	printf("%s\n", errstr);
		return (iio_error (errstr));
	}

	return (iio_status_ok);
}

#define PLZ_MODULE_STATUS_REG_LEN 1
#define PLZ_MODULE_STATUS_REG 1

IIO_STATUS
iio_modbus_plz_open (
			  char * hostname,
			  int * sockp,
			  IIO_MODBUS_PLZINFO * info
			  )
{
	int error = 0;
	char errstr[256];
	short statusReg[PLZ_MODULE_STATUS_REG_LEN];
	if (iio_modbus_connect (hostname, sockp) != iio_status_ok) {
		strcpy (errstr, "iio_modbus_plz_open[iio_modbus_connect()]: ");
		strcat (errstr, iio_emessage_get ());
		return (iio_error (errstr));
	}

	/* Read the module status block */
	if (iio_modbus_read (* sockp, 0, iio_modbus_input_read_fn, 
				PLZ_MODULE_STATUS_REG,
				PLZ_MODULE_STATUS_REG_LEN, statusReg)
			!= iio_status_ok) {
		strcpy (errstr, "iio_modbus_plz_open[iio_modbus_read()]: ");
		strcat (errstr, iio_emessage_get ());
		return (iio_error (errstr));
	}
	if ( (info->oFault = statusReg[0] & 0x01) != 0) {		/* bit 0 */ 
		strcpy(errstr, "iio_modbus_plz_open[status]oFault LED\r\n");
		error = 1;
	}
	if ( (info->iFault = statusReg[0] & 0x02) != 0) {		/* bit 1 */ 
		strcpy(errstr, "iio_modbus_plz_open[status]iFault LED\r\n");
		error = 1;
	}
	if ( (info->fault = statusReg[0] & 0x04) != 0) {		/* bit 2 */ 
		strcpy(errstr, "iio_modbus_plz_open[status]fault LED\r\n");
		error = 1;
	}
	if ( (info->diag = statusReg[0] & 0x08) != 0) {		/* bit 3 */ 
		strcpy(errstr, "iio_modbus_plz_open[status]diag LED\r\n");
		error = 1;
	}
	if ( (info->run = statusReg[0] & 0x10) == 0) {		/* bit 4 */ 
		strcpy(errstr, "iio_modbus_plz_open[status]run LED not lit\r\n");
		error = 1;
	}
	if ( (info->coms = statusReg[0] & 0x20) == 0) {		/* bit 5 */ 
		strcpy(errstr, "iio_modbus_plz_open[status]coms LED not lit\r\n");
		error = 1;
	}
	if (error) {
		strcat (errstr, iio_emessage_get ());
		return (iio_error (errstr));
	}

	return (iio_status_ok);
}



