/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 *
 * mbusplzmc8p.c - plz saftey plc with digital output and inputs 
 *
 *	-host <addr>			IP address of the module
 *
 *  The plc provides access to (3 bytes) 24 inputs and 24 (3 bytes) outputs
 *  as well as access to read the status LEDs
 *
 */

#include <stdio.h>

#include "../internal.h"

#define NUM_CHANNELS 32
#define OUTPUT_START_ADDRESS 1024
#define INPUT_START_ADDRESS 0

    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
	int out_reg_address;
	int in_reg_address;
    IIO_MODBUS_PLZINFO info;
};

struct IIO_MSTATE {
	short outval[NUM_CHANNELS];
};

HIDDEN IIO_STATUS iio_mbusplzmc8p_do(
		IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
		IIO_OP op, unsigned first, unsigned number
		) {
	unsigned int seqno;

	for (seqno = first; seqno < first + number; ++seqno) {
		// printf("seqno = %d\n", seqno); 
		switch (op) {
			case iio_op_write:
				state->outval[seqno] = (uint16_t)iio_data_get(opnode, seqno);
				break;

			case iio_op_readback:
				iio_eret( iio_data_set(opnode, seqno, state->outval[seqno]) );
				break;
			case iio_op_read:
				return iio_error("Operation code not supported by channel");
				break;

			default:
				return iio_error("Operation code not supported by channel");
		}
	}
	iio_eret ( iio_modbus_write (reg->sockfd, 0, reg->out_reg_address + first, number, state->outval));
	return iio_status_ok;
}

HIDDEN IIO_STATUS iio_mbusplzmc8p_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
	unsigned seqno = 0;
	short val = 0;



	switch (op) {
		case iio_op_write:
			return iio_error("Operation code not supported by channel");
			break;

		case iio_op_readback:
			return iio_error("Operation code not supported by channel");

		case iio_op_read:
			iio_eret ( iio_modbus_read (reg->sockfd, 0, iio_modbus_input_read_fn, reg->in_reg_address+first, number, &val));
			for (seqno = first; seqno < first + number; ++seqno){
				iio_eret( iio_data_set(opnode, seqno, val) );
			}
			break;
                        /* seg faults on an all need to investigate but no time now */
		default:
			return iio_error("Operation code not supported by channel");
	}
	return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusplzmc8p_init(IIO_MREG *reg, IIO_MSTATE *state) {
	int i;
	for (i=0; i<NUM_CHANNELS; i++) {
		state->outval[i] = 0;
	}
	iio_eret ( iio_modbus_plz_open (reg->hostname, &reg->sockfd,
				&reg->info));
	return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusplzmc8p_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No inetrnet address specified");
	
	reg->out_reg_address = OUTPUT_START_ADDRESS;
	
	iio_eret( iio_arg(argv, "out_reg_address", iio_arg_int32, &reg->out_reg_address) );

	reg->in_reg_address = INPUT_START_ADDRESS;
	iio_eret( iio_arg(argv, "in_start_address", iio_arg_int32, &reg->in_reg_address) );

	/* register IP channels with the channel list */
	iio_eret(
			iio_chnode(
				module,
				iio_chtype_do, 16, 2,
				iio_mbusplzmc8p_do, NULL
				)
			);
	iio_eret(
			iio_chnode(
				module,
				iio_chtype_di, 16, 2,
				iio_mbusplzmc8p_di, NULL
				)
			);

	return iio_status_ok;
}


IIO_STATUS iio_mbusplzmc8p(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "PLZMC8P",
	    "PLZ saftey plc 24 digital inputs & outputs",
	    "$Revision: 3101 $",
	    iio_multi_yes,
	    iio_mbusplzmc8p_install,
	    iio_mbusplzmc8p_init
	)
    );
    return iio_status_ok;
}
