/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: mbusehc3020.c 413 2007-09-19 07:00:19Z ale077 $
 *
 * mbusehc3020.c - ADVANSYS Modbus EHC3020 High Speed Counter
 *
 *	-host <addr>			IP address of the module
 *
 *      This unit uses 6 registers with data inputing on the
 *      fifth register
 *
 *
 */

#include <stdio.h>
#include <string.h>

#include "../internal.h"

#define INPUT_START_ADDRESS 45461

#define NUM_CHANNELS       1
#define DATA_OFFSET       4


    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    int reg_address;
    IIO_MODBUS_ADVANSYSINFO info;
    unsigned char globalOpMode; // potentially use this for freq vs count
};

HIDDEN IIO_STATUS iio_mbusehc3020_ops(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
	) {
	unsigned int seqno;
	short val[NUM_CHANNELS];
	bzero(val, NUM_CHANNELS*sizeof(short));

	for (seqno = first; seqno < first + number; ++seqno) {
		switch (op) {
			case iio_op_write:
				return iio_error("Operation code not supported by channel");
				break;

			case iio_op_readback:
				return iio_error("Operation code not supported by channel");

			case iio_op_read:
				iio_eret ( iio_modbus_read (reg->sockfd, 0, iio_modbus_input_read_fn, reg->reg_address+DATA_OFFSET+first, number, val));
				iio_eret( iio_data_set(opnode, seqno, val[seqno]) );
				break;

			default:
				return iio_error("Operation code not supported by channel");
		}
	}
	return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusehc3020_init(IIO_MREG *reg, IIO_MSTATE *state) {

    iio_eret ( iio_modbus_advansys_open (reg->hostname, &reg->sockfd,
			    &reg->info));

    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusehc3020_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No internet address specified");

    reg->reg_address = INPUT_START_ADDRESS;
    iio_eret( iio_arg(argv, "reg_address", iio_arg_int32, &reg->reg_address) );

    /* crazy unexplainable advansys thing please let polly know if you work out
     * why this is needed found by trial and error */
    reg->reg_address = crazy_advansys_read_conversion(reg->reg_address);

    /* register IP channels with the channel list */
	iio_eret(
			iio_chnode(
				module,
				iio_chtype_ctr, 1, 1,
				iio_mbusehc3020_ops,
				NULL
				)
			);

    return iio_status_ok;
}


IIO_STATUS iio_mbusehc3020(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "EHC3020",
	    "Advansys EHC3020 6 register 1 counter unit",
	    "$Revision: 413 $",
	    iio_multi_yes,
	    iio_mbusehc3020_install,
	    iio_mbusehc3020_init
	)
    );
    return iio_status_ok;
}
