/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 *
 * mbusavo0200.c - Advansys AVO0200 analog output module (voltage only)
 *
 *	-host <addr>			IP address of the module
 *
 */

#include <stdio.h>
#include <string.h>

#include "../internal.h"

#define OUTPUT_START_ADDRESS 40000

#define NUM_CHANNELS         2
#define VOLTAGE_SCALE        0x7FFF
#define FAIL_TO_ZERO         0x1
#define FAIL_TO_FULL_SCALE   0x5
#define FAIL_TO_LAST_VALUE   0x9


#define CHANNEL_5V         0xA
#define CHANNEL_10V        0xB
#define CHANNEL_INACTIVE   0xC
#define CHANNEL_20mA       0xE

    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    int reg_address; 
    IIO_MODBUS_ADVANSYSINFO info;

    unsigned char globalOpMode;
    unsigned char opMode[NUM_CHANNELS];
};

struct IIO_MSTATE {
    short outval[NUM_CHANNELS];
};

HIDDEN IIO_STATUS iio_mbusavo0200_ops(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

	for (seqno = first; seqno < first + number; ++seqno) {
		switch (op) {
			case iio_op_read:
				return iio_error("Operation code not supported by channel");

			case iio_op_write:
				state->outval[seqno] = (uint16_t)iio_data_get(opnode, seqno); 
				break;

			case iio_op_readback:
				iio_eret( iio_data_set(opnode, seqno, state->outval[seqno]) );
				break;

			default:
				return iio_error("Operation code not supported by channel");
		}
	}
    iio_eret ( iio_modbus_write (reg->sockfd, 0, reg->reg_address, NUM_CHANNELS, state->outval)); 
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusavo0200_init(IIO_MREG *reg, IIO_MSTATE *state) {
    int i;

    for (i=0; i<NUM_CHANNELS; i++) {
	state->outval[i] = 0; 
    }

    iio_eret ( iio_modbus_advansys_open (reg->hostname, &reg->sockfd,
					 &reg->info));
    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusavo0200_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    IIO_CHNODE * chnode;
    int i;
	char * opMode = NULL;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	    return iio_error("No internet address specified");

    reg->reg_address = OUTPUT_START_ADDRESS;
    iio_eret( iio_arg(argv, "reg_address", iio_arg_int32, &reg->reg_address) );
   // printf("reg_address = %d", reg->reg_address);

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_dac, 16, NUM_CHANNELS,
	    iio_mbusavo0200_ops,
	    &chnode
	)
    );

	/* configure channels */
	reg->globalOpMode = CHANNEL_10V;
	iio_eret( iio_arg(argv, "opmode", iio_arg_string, &opMode) );
	if (opMode != NULL) {
		if (strcmp (opMode, "10V") == 0)
			reg->globalOpMode = CHANNEL_10V;
		else if (strcmp (opMode, "5V") == 0)
			reg->globalOpMode = CHANNEL_5V;
		else if (strcmp (opMode, "inactive") == 0)
			reg->globalOpMode = CHANNEL_INACTIVE;
		else if (strcmp (opMode, "20mA") == 0)
			reg->globalOpMode = CHANNEL_20mA;
	}
	for (i=0; i<NUM_CHANNELS; i++) {
		reg->opMode[i] = reg->globalOpMode;
		opMode = NULL;
		iio_eret (iio_arg_index(argv, "opmode", i, iio_arg_string, &opMode));
		if (opMode != NULL) {
			if (strcmp (opMode, "10V") == 0)
				reg->opMode[i] = CHANNEL_10V;
			else if (strcmp (opMode, "5V") == 0)
				reg->opMode[i] = CHANNEL_5V;
			else if (strcmp (opMode, "inactive") == 0)
				reg->opMode[i] = CHANNEL_INACTIVE;
			else if (strcmp (opMode, "20mA") == 0)
				reg->opMode[i] = CHANNEL_20mA;
		}
	}

	for (i=0; i<NUM_CHANNELS; i++)
		switch (reg->opMode[i]) {
			case CHANNEL_10V :
				iio_eret (iio_chnode_linear (chnode, i, 10.0/VOLTAGE_SCALE, 0.0, "V"));
				break;
			case CHANNEL_5V :
				return iio_error("5V not available for the avo0200");
				break;
			case CHANNEL_INACTIVE :
				break;
			case CHANNEL_20mA :
				return iio_error("20V not available for the avo0200");
				break;
			default :
				break;
		}

    return iio_status_ok;
}


IIO_STATUS iio_mbusavo0200(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "AVO0200",
	    "Advansys AVO0200 2 analog outputs",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_mbusavo0200_install,
	    iio_mbusavo0200_init
	)
    );
    return iio_status_ok;
}
