/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: mbusavi1400.c 413 2007-09-19 07:00:19Z ale077 $
 *
 * mbusavi1400.c - ADVANSYS Modbus AVI1400 analog input module
 *
 *	-host <addr>			IP address of the module
 *
 *      This board provides 8 analog inputs with a status for each one so 16
 *      registers are used for each unit
 *
 *
 */

#include <stdio.h>
#include <string.h>

#include "../internal.h"

#define INPUT_START_ADDRESS 45400

#define NUM_CHANNELS       16
#define CHANNEL_5V         0xA
#define CHANNEL_10V        0xB
#define CHANNEL_INACTIVE   0xC
#define CHANNEL_20mA       0xE

#define VOLTAGE_SCALE        0x7FFF
#define CURRENT_SCALE        0xFFFF


    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    IIO_MODBUS_ADVANSYSINFO info;
    int reg_address;
    unsigned char globalOpMode;
    unsigned char opMode[NUM_CHANNELS];
};

HIDDEN IIO_STATUS iio_mbusavi1400_ops(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;
	short val[NUM_CHANNELS];
	bzero(val, NUM_CHANNELS*sizeof(short));

	switch (op) {
		case iio_op_write:
			return iio_error("Operation code not supported by channel");
			break;

		case iio_op_readback:
			return iio_error("Operation code not supported by channel");

		case iio_op_read:
			iio_eret ( iio_modbus_read (reg->sockfd, 0, iio_modbus_input_read_fn, reg->reg_address+first, number, val));
			for (seqno = first; seqno < first + number; ++seqno) {
				iio_eret( iio_data_set(opnode, seqno, val[seqno]) );
			}
			break;

		default:
			return iio_error("Operation code not supported by channel");
	}
	return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusavi1400_init(IIO_MREG *reg, IIO_MSTATE *state) {

    iio_eret ( iio_modbus_advansys_open (reg->hostname, &reg->sockfd,
			    &reg->info));

    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusavi1400_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    IIO_CHNODE * chnode;
    char * opMode = NULL;
    int i;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No internet address specified");

    reg->reg_address = INPUT_START_ADDRESS;
    iio_eret( iio_arg(argv, "reg_address", iio_arg_int32, &reg->reg_address) );

    /* crazy unexplainable advansys thing please let polly know if you work out
     * why this is needed found by trial and error */
    reg->reg_address = crazy_advansys_read_conversion(reg->reg_address);


    reg->globalOpMode = CHANNEL_10V;
    iio_eret( iio_arg(argv, "opmode", iio_arg_string, &opMode) );
    if (opMode != NULL) {
        if (strcmp (opMode, "10V") == 0)
	    reg->globalOpMode = CHANNEL_10V;
	else if (strcmp (opMode, "5V") == 0)
	    reg->globalOpMode = CHANNEL_5V;
	else if (strcmp (opMode, "inactive") == 0)
	    reg->globalOpMode = CHANNEL_INACTIVE;
	else if (strcmp (opMode, "20mA") == 0)
	    reg->globalOpMode = CHANNEL_20mA;
    }
	for (i=0; i<16; i++) {
		reg->opMode[i] = reg->globalOpMode;
		opMode = NULL;
		iio_eret (iio_arg_index(argv, "opmode", i, iio_arg_string, &opMode));
		if (opMode != NULL) {
			if (strcmp (opMode, "10V") == 0)
				reg->opMode[i] = CHANNEL_10V;
			else if (strcmp (opMode, "5V") == 0)
				reg->opMode[i] = CHANNEL_5V;
			else if (strcmp (opMode, "inactive") == 0)
				reg->opMode[i] = CHANNEL_INACTIVE;
			else if (strcmp (opMode, "20mA") == 0)
				reg->opMode[i] = CHANNEL_20mA;
		}
	}

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 16, 16,
	    iio_mbusavi1400_ops,
	    &chnode
	)
    );

    /* configure channels */
    for (i=0; i<16; i++)
        switch (reg->opMode[i]) {
	    case CHANNEL_10V :
	        iio_eret (iio_chnode_linear (chnode, i, 10.0/VOLTAGE_SCALE, 0.0, "V"));
		break;
	    case CHANNEL_5V :
			return iio_error("5V not available for the avi1400");
	        break;
	    case CHANNEL_INACTIVE :
	        break;
	    case CHANNEL_20mA :
	        iio_eret (iio_chnode_linear (chnode, i, 16.0/CURRENT_SCALE, 4.0, "mA"));
	        break;
	    default :
	        break;
	}

    return iio_status_ok;
}


IIO_STATUS iio_mbusavi1400(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "AVI1400",
	    "Advansys AVI1400 16 register 8 analog inputs",
	    "$Revision: 413 $",
	    iio_multi_yes,
	    iio_mbusavi1400_install,
	    iio_mbusavi1400_init
	)
    );
    return iio_status_ok;
}
