/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: mbusadm35010.c 3101 2008-05-16 07:11:40Z roy029 $
 *
 * mbusadm35010.c - Momentum ADM35010 digital I/O module
 *
 *	-host <addr>			IP address of the module
 *
 *      This board provides 16 digital inputs and 16 digital outputs
 *
 *
 */

#include <stdio.h>

#include "../internal.h"

    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    IIO_MODBUS_MOMENTUMINFO info;
    int grayCode;
    int mask;
};

struct IIO_MSTATE {
    short outval;
};

HIDDEN int
iio_enc_gray_to_bin (
		     unsigned short gray,
		     int numBits
		     )
{
    unsigned short tmp;
    unsigned short bin = 0;
    unsigned short lsb = 0;
    char i;
        
    for (i=numBits-1; i >=0; i--) {        /* step through from MSB to LSB */
        tmp = (gray >> i) & 0x01;               
        lsb ^= tmp;
        bin |= lsb << i;
    }
    return (bin);
}

HIDDEN IIO_STATUS iio_mbusadm35010_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;
    short val = 0;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    return iio_error("Operation code not supported by channel");
	    break;

	case iio_op_readback:
	    return iio_error("Operation code not supported by channel");

	case iio_op_read:
	    iio_eret ( iio_modbus_read (reg->sockfd, 0, iio_modbus_read_fn, 0, 1, &val));
	    if (reg->grayCode > 0)
	        val = iio_enc_gray_to_bin (val & reg->mask, reg->grayCode);
	    iio_eret( iio_data_set(opnode, seqno, val) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusadm35010_do(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    return iio_error("Operation code not supported by channel");

	case iio_op_write:
	  
	    state->outval = (uint16_t)iio_data_get(opnode, seqno);
	    iio_eret ( iio_modbus_write (reg->sockfd, 0, 0, 1, &state->outval));
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, state->outval) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusadm35010_init(IIO_MREG *reg, IIO_MSTATE *state) {

    state->outval = 0;

    iio_eret ( iio_modbus_momentum_open (reg->hostname, &reg->sockfd,
					 &reg->info));
    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusadm35010_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    int i;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No inetrnet address specified");
    reg->grayCode = 0;
    reg->mask = 0;
    iio_eret( iio_arg(argv, "graycode", iio_arg_int32, &reg->grayCode) );
    for (i=0; i<reg->grayCode; i++)
        reg->mask |= (1 << i);
    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_di, 16, 1,
	    iio_mbusadm35010_di, NULL
	)
    );
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_do, 16, 1,
	    iio_mbusadm35010_do, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_mbusadm35010(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "ADM35010",
	    "Momentum ADM35010 16/16 digital I/O",
	    "$Revision: 3101 $",
	    iio_multi_yes,
	    iio_mbusadm35010_install,
	    iio_mbusadm35010_init
	)
    );
    return iio_status_ok;
}

