/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: mbusadi35000.c 3101 2008-05-16 07:11:40Z roy029 $
 *
 * mbusadi35000.c - Momentum ADI35000 digital input module
 *
 *	-host <addr>			IP address of the module
 *
 *      This board provides 32 digital inputs
 *
 *
 */

#include <stdio.h>

#include "../internal.h"

    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    IIO_MODBUS_MOMENTUMINFO info;
    int grayCodeGlobal;
    int maskGlobal;
    int grayCode[2];
    int mask[2];
};

struct IIO_MSTATE {
    int tmp;
};

HIDDEN int
iio_enc_gray_to_bin (
		     unsigned short gray,
		     int numBits
		     )
{
    unsigned short tmp;
    unsigned short bin = 0;
    unsigned short lsb = 0;
    char i;
        
    for (i=numBits-1; i >=0; i--) {        /* step through from MSB to LSB */
        tmp = (gray >> i) & 0x01;               
        lsb ^= tmp;
        bin |= lsb << i;
    }
    return (bin);
}

HIDDEN IIO_STATUS iio_mbusadi35000_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;
    short val[2] = {0, 0};

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    return iio_error("Operation code not supported by channel");
	    break;

	case iio_op_readback:
	    return iio_error("Operation code not supported by channel");

	case iio_op_read:
	    iio_eret ( iio_modbus_read (reg->sockfd, 0, iio_modbus_read_fn, 0, 2, val));
	    if (reg->grayCode[seqno] > 0)
	        val[seqno] = iio_enc_gray_to_bin 
		  (val[seqno] & reg->mask[seqno], 
		   reg->grayCode[seqno]);
	    iio_eret( iio_data_set(opnode, seqno, val[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}



HIDDEN IIO_STATUS iio_mbusadi35000_init(IIO_MREG *reg, IIO_MSTATE *state) {

    iio_eret ( iio_modbus_momentum_open (reg->hostname, &reg->sockfd,
					 &reg->info));
    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusadi35000_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    int i, j, graycode = 0;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No inetrnet address specified");
    reg->grayCodeGlobal = 0;
    reg->maskGlobal = 0;
    iio_eret( iio_arg(argv, "graycode", iio_arg_int32, &reg->grayCodeGlobal) );
    for (i=0; i<reg->grayCodeGlobal; i++)
        reg->maskGlobal |= (1 << i);
    for (i=0; i<2; i++) {
        reg->grayCode[i] = reg->grayCodeGlobal;
	reg->mask[i] = reg->maskGlobal;
    }
    for (i=0; i<2; i++) {
        graycode = 0;
        iio_eret (iio_arg_index(argv, "graycode", i, iio_arg_int32, 
				&graycode));
	if (graycode > 0) {
	    reg->grayCode[i] = graycode;
	    reg->mask[i] = 0;
	    for (j=0; j<reg->grayCode[i]; j++)
	        reg->mask[i] |= (1 << j);
	}
    }

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_di, 16, 2,
	    iio_mbusadi35000_di, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_mbusadi35000(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "ADI35000",
	    "Momentum ADI35000 32 digital input",
	    "$Revision: 3101 $",
	    iio_multi_yes,
	    iio_mbusadi35000_install,
	    iio_mbusadi35000_init
	)
    );
    return iio_status_ok;
}
