/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: mbusaao12000.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * mbusaao12000.c - Momentum AAO12000 analog output module
 *
 *	-host <addr>			IP address of the module
 *      -current                        channel wired as a current source
 *
 *      This board provides 4 independent analog voltage/current outputs
 *
 *
 */

#include <stdio.h>
#include <string.h>

#include "../internal.h"

#define NUM_CHANNELS         4
#define VOLTAGE_SCALE        3200
#define CURRENT_SCALE        1600
#define FAIL_TO_ZERO         0x1
#define FAIL_TO_FULL_SCALE   0x5
#define FAIL_TO_LAST_VALUE   0x9

    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    IIO_MODBUS_MOMENTUMINFO info;
    IIO_BOOL globalCurrentMode;
    IIO_BOOL currentMode[NUM_CHANNELS];
    unsigned char globalOpMode;
    unsigned char opMode[NUM_CHANNELS];
    unsigned short configWord;
};

struct IIO_MSTATE {
    short outval[5];
};

HIDDEN IIO_STATUS iio_mbusaao12000_ops(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    return iio_error("Operation code not supported by channel");

	case iio_op_write:
	  
	    state->outval[seqno+1] = (uint16_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, state->outval[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    iio_eret ( iio_modbus_write (reg->sockfd, 0, 0, 5, state->outval));
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusaao12000_init(IIO_MREG *reg, IIO_MSTATE *state) {
    int i;

    reg->configWord = 0;
    for (i=0; i<4; i++) {
        reg->configWord |= (reg->opMode[i] << (4*i));
	state->outval[i+1] = 0;
    }
    state->outval[0] = reg->configWord;
    iio_eret ( iio_modbus_momentum_open (reg->hostname, &reg->sockfd,
					 &reg->info));
    iio_eret ( iio_modbus_write (reg->sockfd, 0, 0, 5, state->outval));
    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusaao12000_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    IIO_CHNODE * chnode;
    int i;
    char * opMode = NULL;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No inetrnet address specified");

    reg->globalOpMode = FAIL_TO_ZERO;
    iio_eret( iio_arg(argv, "opmode", iio_arg_string, &opMode) );
    if (opMode != NULL) {
        if (strcmp (opMode, "fail_to_zero") == 0)
	    reg->globalOpMode = FAIL_TO_ZERO;
	else if (strcmp (opMode, "fail_to_full_scale") == 0)
	    reg->globalOpMode = FAIL_TO_FULL_SCALE;
	else if (strcmp (opMode, "fail_to_last_value") == 0)
	    reg->globalOpMode = FAIL_TO_LAST_VALUE;
    }
    for (i=0; i<4; i++)
        reg->opMode[i] = reg->globalOpMode;
    for (i=0; i<4; i++) {
        opMode = NULL;
        iio_eret (iio_arg_index(argv, "opmode", i, iio_arg_string, &opMode));
	if (opMode != NULL) {
	    if (strcmp (opMode, "fail_to_zero") == 0)
	        reg->opMode[i] = FAIL_TO_ZERO;
	    else if (strcmp (opMode, "fail_to_full_scale") == 0)
	        reg->opMode[i] = FAIL_TO_FULL_SCALE;
	    else if (strcmp (opMode, "fail_to_last_value") == 0)
	        reg->opMode[i] = FAIL_TO_LAST_VALUE;
	}
    }

    reg->globalCurrentMode = iio_bool_false;
    for (i=0; i<4; i++) {
        reg->currentMode[i] = iio_bool_false;
    }
    iio_eret( iio_arg(argv, "current", iio_arg_bool, &reg->globalCurrentMode) );
    if (reg->globalCurrentMode)
        for (i=0; i<4; i++)
	    reg->currentMode[i] = iio_bool_true;
    for (i=0; i<4; i++)
        iio_eret (iio_arg_index(argv, "current", i, iio_arg_bool, &(reg->currentMode[i])));

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_dac, 16, 4,
	    iio_mbusaao12000_ops,
	    &chnode
	)
    );

    /* configure channels */
    for (i=0; i<4; i++)
        if (reg->currentMode[i]) {
	    iio_eret (iio_chnode_linear (chnode, i, 1.0/1600, 0.0, "mA"));
	} else {
	    iio_eret (iio_chnode_linear (chnode, i, 1.0/3200, 0.0, "V"));
	}

    return iio_status_ok;
}


IIO_STATUS iio_mbusaao12000(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "AAO12000",
	    "Momentum AAO12000 4 analog outputs",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_mbusaao12000_install,
	    iio_mbusaao12000_init
	)
    );
    return iio_status_ok;
}
