/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: mbusaai14000.c 3101 2008-05-16 07:11:40Z roy029 $
 *
 * mbusaai14000.c - Momentum AAI14000 analog input module
 *
 *	-host <addr>			IP address of the module
 *
 *      This board provides 16 analog inputs
 *
 *
 */

#include <stdio.h>
#include <string.h>

#include "../internal.h"

#define NUM_CHANNELS       16
#define CHANNEL_5V         0xA
#define CHANNEL_10V        0xB
#define CHANNEL_INACTIVE   0xC
#define CHANNEL_20mA       0xE

    /* register pointer structure */
struct IIO_MREG {
    char *hostname;     /* internet address of the module */
    int sockfd;
    IIO_MODBUS_MOMENTUMINFO info;
    unsigned char globalOpMode;
    unsigned char opMode[NUM_CHANNELS];
};

HIDDEN IIO_STATUS iio_mbusaai14000_ops(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;
    short val = 0;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    return iio_error("Operation code not supported by channel");
	    break;

	case iio_op_readback:
	    return iio_error("Operation code not supported by channel");

	case iio_op_read:
	    iio_eret ( iio_modbus_read (reg->sockfd, 0, iio_modbus_read_fn, seqno, 1, &val));
	    iio_eret( iio_data_set(opnode, seqno, val) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_mbusaai14000_init(IIO_MREG *reg, IIO_MSTATE *state) {
    short configWord[16];
    int i, j;

    for (i=0; i<4; i++) {
        configWord[i] = 0;
	for (j=0; j<4; j++)
	    configWord[i] |= (reg->opMode[4*i+j] << (4*j));
    }
    for (i=4; i<16; i++)
        configWord[i] = 0;

    iio_eret ( iio_modbus_momentum_open (reg->hostname, &reg->sockfd,
					 &reg->info));
    iio_eret ( iio_modbus_write (reg->sockfd, 0, 0, 16, configWord));
    return (iio_status_ok);
}


HIDDEN IIO_STATUS iio_mbusaai14000_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    IIO_CHNODE * chnode;
    char * opMode = NULL;
    int i;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* decode the arguments */
    /* get the mandatory internet address */
    reg->hostname = NULL;
    iio_eret( iio_arg(argv, "address", iio_arg_string, &reg->hostname) );
    if (reg->hostname == NULL)
	return iio_error("No inetrnet address specified");

    reg->globalOpMode = CHANNEL_10V;
    iio_eret( iio_arg(argv, "opmode", iio_arg_string, &opMode) );
    if (opMode != NULL) {
        if (strcmp (opMode, "10V") == 0)
	    reg->globalOpMode = CHANNEL_10V;
	else if (strcmp (opMode, "5V") == 0)
	    reg->globalOpMode = CHANNEL_5V;
	else if (strcmp (opMode, "inactive") == 0)
	    reg->globalOpMode = CHANNEL_INACTIVE;
	else if (strcmp (opMode, "20mA") == 0)
	    reg->globalOpMode = CHANNEL_20mA;
    }
    for (i=0; i<16; i++)
        reg->opMode[i] = reg->globalOpMode;
    for (i=0; i<16; i++) {
        opMode = NULL;
        iio_eret (iio_arg_index(argv, "opmode", i, iio_arg_string, &opMode));
	if (opMode != NULL) {
	    if (strcmp (opMode, "10V") == 0)
	        reg->opMode[i] = CHANNEL_10V;
	    else if (strcmp (opMode, "5V") == 0)
	        reg->opMode[i] = CHANNEL_5V;
	    else if (strcmp (opMode, "inactive") == 0)
	        reg->opMode[i] = CHANNEL_INACTIVE;
	    else if (strcmp (opMode, "20mA") == 0)
	        reg->opMode[i] = CHANNEL_20mA;
	}
    }

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 16, 16,
	    iio_mbusaai14000_ops,
	    &chnode
	)
    );

    /* configure channels */
    for (i=0; i<16; i++)
        switch (reg->opMode[i]) {
	    case CHANNEL_10V :
	        iio_eret (iio_chnode_linear (chnode, i, 1.0/3200.0, 0.0, "V"));
		break;
	    case CHANNEL_5V :
	        iio_eret (iio_chnode_linear (chnode, i, 1.0/6400.0, 0.0, "V"));
	        break;
	    case CHANNEL_INACTIVE :
	        break;
	    case CHANNEL_20mA :
	        iio_eret (iio_chnode_linear (chnode, i, 1.0/2000.0, 4.0, "mA"));
	        break;
	    default :
	        break;
	}

    return iio_status_ok;
}


IIO_STATUS iio_mbusaai14000(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "AAI14000",
	    "Momentum AAI14000 16 digital inputs",
	    "$Revision: 3101 $",
	    iio_multi_yes,
	    iio_mbusaai14000_install,
	    iio_mbusaai14000_init
	)
    );
    return iio_status_ok;
}
