/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: isapc.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * isapc.c -- module driver for a generic ISA PC (PC/AT)
 *
 * This module is to support Industry Standard Architecture (ISA) PCs. It 
 * provides a single ISA-bus channel. The channel provides address resolution
 * for the ISA memory space (top of local memory to to 16M). Note ISA IO space
 * can't be accessed on these machines, except through the 80x86-style INP and
 * OUTP instructions; drivers that use these instruction cannot be portable,
 * and so are not part of IIO.
 *
 * This driver can be extended for EISA (32-bit addresses) as soon as I can
 * be sure about the addresses translations (and get such a machine and some
 * boards to test things).
 *
 * There is one module option, -mbytes, which should be set to the amount of
 * local memory on the machine, in megabytes.
 */
#include "../internal.h"


struct IIO_MREG {
    /* address of top of local memory */
    uint32_t memtop;
};

HIDDEN IIO_STATUS iio_isapc_isa(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Address resolutions. There is no offset between processor physical
     * addresses and bus addresses. However, low addresses are occluded
     * by local memory
     */
    uint32_t addr = (uint32_t)iio_data_get_addr(opnode, first);
    /* IIO_SIZE size = (IIO_SIZE)(op & IIO_SIZE_MASK); */

    switch (op & IIO_SPACE_MASK) {
    case iio_space_mem:
    case iio_space_mem24:
	if (addr > 0xffffff)
	    return iio_error("Address out of range");
	if (addr < reg->memtop)
	    return iio_error("Address within local memory");
	break;

    case iio_space_port:
	if (addr > 0xffff)
	    return iio_error("Address out of range");
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }

    /* write back result */
    iio_eret( iio_data_set_addr(opnode, first, (void *)addr) );
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_isapc_init(IIO_MREG *reg, IIO_MSTATE *state) {

    /* The driver needs no initialisation (yet) */
    return iio_status_ok;
}


IIO_STATUS iio_isapc_install(IIO_MODULE *module, char *argv[]) {

    IIO_MREG *reg;
    uint8_t mbytes = 0;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* work out the lower memory limit */
    iio_eret( iio_arg(argv, "mbytes", iio_arg_channel, &mbytes) );
    reg->memtop = (mbytes << 20);

    /* register one ISA channel */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_isa, 0, 1,
	    iio_isapc_isa, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_isapc(void) {
    iio_eret(
	iio_minfo(
	    "isapc",			/* Our module ID */
	    "ISA PC",			/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_no,		/* we can be multiply installed */
	    iio_isapc_install,		/* the install function */
	    iio_isapc_init		/* the init function */
	)
    );
    return iio_status_ok;
}
