/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipwatchdog.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipwatchdog.c -- driver for GreenSpring IP-Watchdog
 *
 * The IP-Watchdog provides 8 digital IO lines, a programmable periodic or
 * watchdog timer/interrupter, external outputs for VMEbus SYSRESET, four
 * power supply monitors (+5V, +12V, -12V and a selectable +/- 24V, +/- 48V),
 * and a programmable temperature monitor/thermostat.
 * 
 * Module options:
 *
 *	-slot <channel>			IP slot
 *
 * Currently, only the digital IO and the monitors are implemented in this
 * driver. Interrupts from any source are not possible, until the rest of
 * IIO gets this capability. 
 */
#include "../internal.h"
#include "../chip/ds1620.h"


    /* Timer control register bit values */
#define TIM_CONTROL_LOAD	(1 << 0)
#define TIM_CONTROL_RUN		(1 << 1)
#define TIM_CONTROL_PERIODIC	(1 << 2)
#define TIM_CONTROL_ARMED	(1 << 3)
#define TIM_CONTROL_RESET	(1 << 4)

    /* Monitor output polarity register bit values */
#define MON_POLARITY_P05	(1 << 0)
#define MON_POLARITY_P12	(1 << 1)
#define MON_POLARITY_N12	(1 << 2)
#define MON_POLARITY_TIM	(1 << 3)
#define MON_POLARITY_AUX	(1 << 4)
#define MON_POLARITY_THERM	(1 << 5)

    /* Monitor data register bit values */
#define MON_DATA_P05		(1 << 0)
#define MON_DATA_P12		(1 << 1)
#define MON_DATA_N12		(1 << 2)
#define MON_DATA_AUX		(1 << 4)
#define MON_DATA_THERM		(1 << 5)

    /* Interrupt register bit values (except vector register) */
#define INT_DI7			(1 << 15)
#define INT_DI6			(1 << 14)
#define INT_DI5			(1 << 13)
#define INT_DI4			(1 << 12)
#define INT_DI3			(1 << 11)
#define INT_DI2			(1 << 10)
#define INT_DI1			(1 <<  9)
#define INT_DI0			(1 <<  8)
#define	INT_THERM		(1 <<  5)
#define	INT_AUX			(1 <<  4)
#define	INT_TIM			(1 <<  3)
#define	INT_N12			(1 <<  2)
#define	INT_P12			(1 <<  1)
#define	INT_P05			(1 <<  0)


    /* register pointer structure */
struct IIO_MREG {
    IIO slot;			/* our IP slot */

    /* registers by name */
    volatile uint16_t
	*io_data,		/* IO output data/readback (IO) */
	*io_indata,		/* IO input data (RB) */
	*mon_polarity,		/* Voltage Monitor output polarity (MOP) */
	*mon_data,		/* Voltage Monitor data output (MOV) */
	*tim_load,		/* Timer preload (TIL) */
	*tim_count,		/* Timer count (TIV) */
	*tim_control,		/* Timer control (TIC) */
	*therm_data,		/* Thermometer data (THD) */
	*therm_reset,		/* Thermometer reset (THC) */
	*int_data,		/* Interrupt output (INTD) */
	*int_polarity,		/* Interrupt polarity (INTPOL) */
	*int_enable,		/* Interrupt enable (INTEN) */
	*int_clear,		/* Interrupt clear (INTCLR) */
	*int_pending,		/* Interrupt pending (INTPEN) */
	*int_vector;		/* Interrupt user vector (VECTOR) */

    /* temperature limits */
    int th, tl;
};


HIDDEN IIO_STATUS iio_ipwatchdog_ocio(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * There is one 8-bit DIO port, with open collector outputs with
     * optional pull-up, linked to an 8-bin digital input port. Although
     * an 8-bit port, we use 16-bit accesses
     */
    switch (op) {
    case iio_op_write:
	*reg->io_data = ~(uint8_t)iio_data_get(opnode, 0);
	break;

    case iio_op_readback:
	iio_eret( iio_data_set(opnode, 0, ~*reg->io_data & 0xff) );
	break;

    case iio_op_read:
	iio_eret( iio_data_set(opnode, 0, *reg->io_indata & 0xff) );
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipwatchdog_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * There is one 8-bit DI port, which is actually the monitor voltage,
     * timer and thermal overrange register. The two top bits are unused
     * and are masked out. The bits are thus
     *
     *		0	P05	+5V out of range
     *		1	P12	+12V out of range
     *		2	N12	-12V out of range
     *		3	TI	timer zero
     *		4	AX	auxiliary voltage out of range
     *		5	TH	temperature out of range
     */
    switch (op) {
    case iio_op_read:
	iio_eret( iio_data_set(opnode, 0, *reg->mon_data & 0x3f) );
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN int iio_ipwatchdog_ds1620(IIO_MREG *reg, IIO_DS1620_OP op, int *data) {
    /* 
     * Access callback function required by the DS1620 chip driver.
     * The gets or set the 1-bit registers on the IP which connect to
     * the chip itself, according to <op> 
     */
    switch (op) {
    case iio_ds1620_setrst:
	*reg->therm_reset = 0x1;
	return 0;

    case iio_ds1620_clrrst:
	*reg->therm_reset = 0x0;
	return 0;

    case iio_ds1620_getbit:
	*data = (int)*reg->therm_data & 0x1;
	return 0;

    case iio_ds1620_putbit:
	*reg->therm_data = (uint16_t)(*data & 0x1);
	return 0;
    }
    return -1;
}


HIDDEN IIO_STATUS iio_ipwatchdog_adc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * There is one 9-bit ADC which reads the ambient temperature.
     * The ADC is read serially, using the DS1620 chip driver in 
     * conjuction with iio_ipwatchdog_ds1620() above
     */
    int temp;

    switch (op) {
    case iio_op_read:
	iio_ds1620_temp_read(
	    (IIO_DS1620_ACFN)iio_ipwatchdog_ds1620,
	    (void *)reg, &temp
	);
	iio_eret( iio_data_set(opnode, 0, temp) );
	break;

    case iio_op_show:
	iio_ds1620_show(NULL, NULL);
	iio_ds1620_show(
	    (IIO_DS1620_ACFN)iio_ipwatchdog_ds1620,
	    (void *)reg
	);
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipwatchdog_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Since the device has readback registers, there is no need for
     * a state structure.
     *
     * After writing to the thermometer chip, there should be a delay for 10ms
     * to allow the chips non-volatile memory to be set
     */

    /* check my identity */
    iio_eret( iio_ipinfo_ident(reg->slot, 0xf0, 0x54) );

    /* put the thermostat into CPU mode */
    iio_eret(
	iio_ds1620_cr_write(
	    (IIO_DS1620_ACFN)iio_ipwatchdog_ds1620,
	    (void *)reg,
	    IIO_DS1620_CR_CPU
	)
    );
    iio_roughdelay(10);

    /* set the high and low thermostat temperatures */
    iio_eret(
	iio_ds1620_th_write(
	    (IIO_DS1620_ACFN)iio_ipwatchdog_ds1620,
	    (void *)reg,
	    reg->th
	)
    );
    iio_roughdelay(10);
    iio_eret(
	iio_ds1620_tl_write(
	    (IIO_DS1620_ACFN)iio_ipwatchdog_ds1620,
	    (void *)reg,
	    reg->tl
	)
    );
    iio_roughdelay(10);

    /* start continuous conversions (is this superfluous?) */
    iio_eret(
	iio_ds1620_conv_start(
	    (IIO_DS1620_ACFN)iio_ipwatchdog_ds1620,
	    (void *)reg
	)
    );

#if 0
    iio_ds1620_show(NULL, NULL);
    iio_ds1620_show((IIO_DS1620_ACFN)iio_ipwatchdog_ds1620, (void *)reg);
#endif

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipwatchdog_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    IIO_CHNODE *chnode;
    double th = 50.0, tl = 5.0;

    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get the slot argument */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );
    if (!reg->slot)
	return iio_error("No IP slot specified");

    /* thermostat temperatures */
    iio_eret( iio_arg(argv, "temphi", iio_arg_double, &th) );
    iio_eret( iio_arg(argv, "templo", iio_arg_double, &tl) );
    if ((th > 125.0) || (th < -55.0))
	return iio_error("-temphi argument out of range");
    if ((tl > 125.0) || (tl < -55.0))
	return iio_error("-templo argument out of range");
    if (th < tl)
	return iio_error("-templo argument exceeds -temphi");
    reg->th = (int)(th * 2 + 0.5);
    reg->tl = (int)(tl * 2 + 0.5);

    /* resolve local register addresses to virtual */
    iio_eret( 
	iio_resolve_list(
	    reg->slot, iio_space_io,
	    iio_size_16, 0x00, &reg->io_data,
	    iio_size_16, 0x02, &reg->io_indata,
	    iio_size_16, 0x04, &reg->mon_polarity,
	    iio_size_16, 0x06, &reg->mon_data,
	    iio_size_16, 0x08, &reg->tim_load,
	    iio_size_16, 0x0a, &reg->tim_count,
	    iio_size_16, 0x0c, &reg->tim_control,
	    iio_size_16, 0x0e, &reg->therm_data,
	    iio_size_16, 0x10, &reg->therm_reset,
	    iio_size_16, 0x12, &reg->int_data,
	    iio_size_16, 0x14, &reg->int_polarity,
	    iio_size_16, 0x16, &reg->int_enable,
	    iio_size_16, 0x18, &reg->int_clear,
	    iio_size_16, 0x1a, &reg->int_pending,
	    iio_size_16, 0x1c, &reg->int_vector,
	    0
	)
    );

    /* the 8-bit general-purpose DIO port */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_ocio, 8, 1,
	    iio_ipwatchdog_ocio, NULL
	    )
	);

    /* the 8-bit voltage-monitor DI port */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_di, 8, 1,
	    iio_ipwatchdog_di, NULL
	    )
	);

    /* the 9-bit thermometer ADC */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 9, 1,
	    iio_ipwatchdog_adc, &chnode
	    )
	);
    iio_eret(
	iio_chnode_linear( chnode, 0, 0.5, 0.0, "C")
    );

    return iio_status_ok;
}


IIO_STATUS iio_ipwatchdog(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"ipwatchdog",
	"GreenSpring IP-Watchdog",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_ipwatchdog_install,
	iio_ipwatchdog_init
    );
}
