/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipservo.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipservo.c -- driver for GreenSpring IP-Servo
 * Robin Kirkham, October 1996, from Mr. Plod
 *
 * The GreenSpring IP-Servo provides twin LM628 closed-loop position or
 * velocity servo chips, each with incremental encoder inputs and 12-bit
 * +/-5V range DAC outputs, which connect to the velocity drive of the servo.
 *
 * This module uses the LM628 chip driver to do most of the work; it is
 * really a mini-module driver. Only the IP-Servo-specific stuff is handled
 * here
 *
 * Module options:
 *
 *	-slot <channel>			IP slot
 *	-4mhz/-no-4mhz 			indicates a 4MHz clock IP (rare)
 *
 */
#include <math.h>
#include "../internal.h"
#include "../chip/lm628.h"

#define NCHAN 2

    /* register pointer structure */
struct IIO_MREG {
    IIO slot;				/* our IP slot */
    uint8_t *ivect;			/* interrupt vector register (both) */
    IIO_LM628_MREG lm628[2];		/* LM628 chip driver register */
};

    /* module state structure */
struct IIO_MSTATE {
    IIO_LM628_MSTATE lm628[2];		/* LM628 chip driver state */
};


HIDDEN IIO_STATUS iio_ipservo_sc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Operate on the LM628 chips. We simply pass the request straight
     * on to the LM628 chip driver. Note that we have to pass the
     * opnode/seqno on as well so the chip driver can get/set its data
     */
    int seqno;

    /* operate on each chip separately */
    for (seqno = first; seqno < first + number; ++seqno) {
	iio_eret(
	    iio_lm628_sc(
		&state->lm628[seqno],
		&reg->lm628[seqno],
		opnode, op, seqno
	    )
	);
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipservo_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialise the module using the configuration already loaded in the
     * state and register structures. 
     */
    int seqno;
#if 0
    /* check the IP module's identity */
    iio_eret( iio_ipinfo_ident(reg->slot, 0xf0, 0x1c) );
#endif

    /* initialise each chip separately */
    for (seqno = 0; seqno < NCHAN; ++seqno) {
	iio_eret(
	    iio_lm628_init(
		&state->lm628[seqno],
		&reg->lm628[seqno]
	    )
	);
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipservo_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the IP base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    IIO_CHNODE *chnode;
    IIO_BOOL mhz4 = iio_bool_false;
    int seqno;

    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* get the slot argument */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );
    if (!reg->slot)
	return iio_error("No slot specified");

    /* check if the chip clock is 4Mhz; 8Mhz is default */
    iio_eret( iio_arg(argv, "4mhz", iio_arg_bool, &mhz4) );

    /* register two channels with the channel list */
    iio_eret(
        iio_chnode(
            module,
            iio_chtype_sc, 32, NCHAN,
            iio_ipservo_sc, 
            &chnode
        )
    );

    /* do each chip/channel separately */
    for (seqno = 0; seqno < NCHAN; ++seqno) {
	volatile uint8_t *csr, *dr;
	IIO_BOOL bipolar = iio_bool_true;

	/*
	 * Compute the chip register addresses (CSR and DR).
	 * Yes, the interrupt vector get done twice
	 */
        iio_eret(
            iio_resolve_list(
                reg->slot, iio_space_io,
                iio_size_8, 0x4 * seqno + 0x1, &csr,
                iio_size_8, 0x4 * seqno + 0x3, &dr,
                iio_size_8, 0x9, &reg->ivect,
                0
            )
        );

	/*
	 * Look for the bipolar option flag, which indicates the DAC output
	 * range (-5..5V, or 0..5V). The IP-Servo is always 12-bits
	 */
        iio_eret(
	    iio_arg_index(argv, "bipolar", seqno, iio_arg_bool, &bipolar)
	);

	/*
	 * Install the chip driver. This fills out its part of the register
	 * structure, parses the rest of the arguments and fixes up the channel
	 * information. The seqno is used only to index the arguments, and
	 * to call the chnode information functions
	 */
	iio_eret(
	    iio_lm628_install(
		&reg->lm628[seqno], csr, dr,
		mhz4 ? 4000000 : 8000000,
		iio_bool_true, bipolar ? 10.0 : 5.0,
		argv, chnode, seqno
	    )
	);
    }
    return iio_status_ok;
}


IIO_STATUS iio_ipservo(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"ipservo",
	"GreenSpring IP-Servo",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_ipservo_install,
	iio_ipservo_init
    );
}
