/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipserial.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipserial.c -- driver for GreenSpring IP-Serial
 *
 * This is not a driver per se, but a wrapper that calls operating system
 * magic to install a kernel device driver for the IP-Serial. All it does
 * is parse some options and resolve the chip register addresses, then
 * calls iio_exec()
 *
 * Module options:
 *
 *	-slot <channel>			IP slot
 *	-init <prog>			execute the program to install
 *	-argument.0 <args>		arguments for init prog, channel A
 *	-argument.1 <args>		arguments for init prog, channel B
 */
#include "../internal.h"


    /* register pointer structure */
struct IIO_MREG {
    IIO slot;				/* our IP slot */
    char *init;				/* external init porgram */

    /* per serial channel */
    struct {
	volatile uint8_t *cr, *dr;	/* 85C30 chip register pointers */
	char *args;			/* argument string passed to init */
    } zs[2];
};


HIDDEN IIO_STATUS iio_ipserial_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Check the identity of the IP, then call the functions to ...
     */
    unsigned port;

    /* check my identity */
    iio_eret( iio_ipinfo_ident(reg->slot, 0xf0, 0x10) );

    /* execute the external installation program for each channel */
    if (reg->init)
	for (port = 0; port < 2; ++port) {
	    iio_eret(
		iio_exec(
		    "%s 0x%x 0x%x %s", 
		    reg->init,
		    (unsigned)reg->zs[port].cr,
		    (unsigned)reg->zs[port].dr,
		    reg->zs[port].args
		)
	    );
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipserial_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the IP-specific arguments, create a register structure, and
     * and resolve register addresses and options into it
     */
    IIO_MREG *reg;
    unsigned port;

    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get the slot argument */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );
    if (!reg->slot)
	return iio_error("No IP slot specified");

    /* get the init program argument */
    iio_eret( iio_arg(argv, "init", iio_arg_string, &reg->init) );
    if (!reg->init)
	return iio_error("No init program specified");

    /* get the optional argument strings */
    for (port = 0; port < 2; ++port) {
	reg->zs[port].args = "";
	iio_eret(
	    iio_arg_index(
		argv, "argument", port,
		iio_arg_string, &reg->zs[port].args
	    )
	);
    }


    /* resolve the four register addresses to PHYSICAL (not virtual) */
    iio_eret(
	iio_resolve_physical(
	    reg->slot, iio_space_io, iio_size_8,
	    5, (void **)&reg->zs[0].cr
	)
    ); 
    iio_eret(
	iio_resolve_physical(
	    reg->slot, iio_space_io, iio_size_8,
	    7, (void **)&reg->zs[0].dr
	)
    ); 
    iio_eret(
	iio_resolve_physical(
	    reg->slot, iio_space_io, iio_size_8, 
	    1, (void **)&reg->zs[1].cr
	)
    ); 
    iio_eret(
	iio_resolve_physical(
	    reg->slot, iio_space_io, iio_size_8,
	    3, (void **)&reg->zs[1].dr
	)
    ); 

    /* nothing else to do here */
    return iio_status_ok;
}


IIO_STATUS iio_ipserial(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "ipserial",
	    "GreenSpring IP-Serial",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_ipserial_install,
	    iio_ipserial_init
	)
    );
    return iio_status_ok;
}
