/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipquadrature.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipquadrature.c -- driver for GreenSpring IP-Quadrature
 *
 * Jonathon Ralston, December 1997
 * CSIRO Division of Exploration and Mining
 *
 * The GreenSpring IP-Quadrature provides four independent 24-bit
 * quadrature decoder channels. Channels may also be used as general
 * purpose up/count counters.  Inputs are configured as either
 * single-ended (TTL/CMOS) or differential (RS-422) by removing the
 * appropriate 120 ohm resistive terminator on the IndustryPack.
 *
 * The hardware counter range of the IP-Quadrature has been extended
 * in software from 24 to 32 bits using basic Nyquist criteria.  The
 * encoder value is stored in two 32 bit integers: enc_counter and
 * ext_counter.
 *
 * The sign, borrow, and carry bits are currently ignored
 */


#if 0
#define IIO_IPQUAD_DEBUG           /* Comment out debug flag for final version */
#endif

#include "../internal.h"

#define QUAD_CHAN             4    /* Number of IP-Quadrature channels */
#define QUAD_COUNT_MAX 16777216    /* Maximum quadrature hardware count (24 bits) */

#define MANUFACTURER_ID    0xf0
#define MODEL_NUMBER       0x41

/* reset and enable operations  */

#define CSR_RESET_TTL      0xf9    /* TTL termination for channels 1 and 2 */
#define CSR_RESET_RS422    0xf8    /* NB: TTL terminator must be removed from IP-Quad board */
#define CR_RESET           0x20
#define ICR_RESET          0x48
#define OCR_SET            0x90
#define OCR_SET_DIVN       0x94    /* Divide by N mode */
#define QR_COUNT_ENABLE    0xc0
#define QR_COUNT_X1        0xc1
#define QR_COUNT_X2        0xc2
#define QR_COUNT_X4        0xc3
#define QR_QUAD_ENABLE     0xc1
#define CHCR_RESET         0x80

/* read/write operations */

#define CR_READ            0x03
#define CR_WRITE           0x05   /* clear borrow/carry on counter write/reset */
#define CR_COUNTER_SET     0x08
#define CR_BCS_CLEAR       0x04   /* clear borrow/carry/sign flags */

/* Prototypes */

IIO_STATUS iio_ipquad( void );

HIDDEN IIO_STATUS iio_ipquadrature_init(IIO_MREG *reg, IIO_MSTATE *state);

HIDDEN IIO_STATUS iio_ipquadrature_install(IIO_MODULE *module, char *argv[]);

HIDDEN IIO_STATUS iio_ipquadrature_op( IIO_MSTATE *state, IIO_MREG *reg, 
				 IIO_OPNODE *opnode,  IIO_OP op, unsigned first, unsigned number );


/* 
 * Register pointer structure
 */

struct IIO_MREG {
  
  IIO      slot;          /* IP slot */
  uint8_t  prescaler;     /* X1, X2, or X4 quadrature prescaler */

  /* 
   * Channel latches and registers.  Register nomenclature follows the
   * IP-Quadrature user manual.  
   */

  struct {
    volatile uint8_t

    *ol,               /* counter output latch */
    *pcr,              /* preset/compare register */
    *sr,               /* status register */
    *cr,               /* counter control register */
    *icr,              /* input control register */
    *ocr,              /* input control register */
    *qr,               /* quadrature register */
    *chcr;             /* channel configuration register */

  } enc[QUAD_CHAN];     /* quadrature decoder device */

  volatile uint8_t
    *csr,                       /* control and status register */
    *vr;			/* (interrupt) vector register */
  
};


/*
 * IP Quadrature state data structure 
 */

struct IIO_MSTATE {
  
  int32_t  enc_counter[QUAD_CHAN];       /* Last encoder counter */
  int32_t  ext_counter[QUAD_CHAN];       /* Domain extension counter */

};



/*
 * IP-Quadrature operation function
 *
 * Valid options are iio_op_read, iio_op_write, iio_op_clear.
 *
 */

HIDDEN IIO_STATUS iio_ipquadrature_op( IIO_MSTATE *state, IIO_MREG *reg, 
  IIO_OPNODE *opnode,  IIO_OP op, unsigned first, unsigned number ) 
{
  
  int      chan;
  int32_t  val, pd_val;  
  uint8_t  byte;
  
  switch( op ) {
    
  case( iio_op_clear ):  
    
    /* 
     * Reset the IP-Quadrature. Set into standard polarity counting mode.
     */

    
    for( chan=0; chan < QUAD_CHAN; chan++ ) {
      
      *reg->enc[chan].cr   = CR_RESET;              /* Master reset */
      *reg->enc[chan].icr  = ICR_RESET;             /* Input control register */    
      *reg->enc[chan].qr   = reg->prescaler;        /* Set quadrature prescaler */
      *reg->enc[chan].ocr  = OCR_SET;               /* Output control register */
      *reg->enc[chan].chcr = CHCR_RESET;            /* Channel configuration register */    
      *reg->enc[chan].cr   = CR_BCS_CLEAR;          /* Counter control register */     
    }
    
#ifdef RS422
    *reg->csr             = CSR_RESET_RS422;         /* CS register for RS422 input */
#else
    *reg->csr             = CSR_RESET_TTL;           /* CS register for TTL input */
#endif
    
    /*
     * Reset encoder and domain extension counters
     */
    
    for( chan=0; chan < QUAD_CHAN; chan++ ) {
      state->ext_counter[chan] = 0;
      state->enc_counter[chan] = 0;
    }
    break;
    
    
  case( iio_op_read ):  
    
    /* 
     * Read the quadrature encoder counter.  The counter values must
     * be first transfered to the output latch register, as the
     * counter register is not directly readable.
     */
    
    for( chan = first; chan < first + number; ++chan ) {
      
      /* Load the counter output latch */
      
      *reg->enc[chan].cr  = CR_READ;             /* Counter control register */      
      
      /* 
       * Read in the 24-bit counter value from the output latch in
       * three bytes, LSB first. 
       */
      
      byte = *reg->enc[chan].ol;                /* LSB */
      val  = byte;                         
      
      byte = *reg->enc[chan].ol;
      val |= byte << 8;                      
      
      byte = *reg->enc[chan].ol;
      val |= byte << 16;

      /* Test to see if a boundary has been crossed since last time.
       * Here we invoke a basic Nyquist sampling theory assumption
       * where the extended domain counter is incremented/decremented
       * if the difference between the current and last encoder values
       * exceeds 2^23.  
       */
      
      if( (val - state->enc_counter[chan]) > QUAD_COUNT_MAX/2 ) {
	state->ext_counter[chan] -= QUAD_COUNT_MAX;                      /* Borrow */

#ifdef IIO_IPQUAD_DEBUG
	printf( "Borrow...\n" );
#endif
      }
      else if( (val - state->enc_counter[chan]) < -QUAD_COUNT_MAX/2 ) {
	state->ext_counter[chan] += QUAD_COUNT_MAX;                      /* Carry */

#ifdef IIO_IPQUAD_DEBUG
	printf( "Carry...\n" );
#endif
      }
      

      /* Save encoder value for next read operation  */ 
            
      state->enc_counter[chan] = val;
      
#ifdef IIO_IPQUAD_DEBUG  
      fprintf( stderr, "v:e = %d : %d\n", val, state->ext_counter[chan] );
#endif
     
      /*
       * Send result to iio, with appropriate counter domain extension added...
       */

      iio_eret( iio_data_set( opnode, chan, (int32_t)(val + state->ext_counter[chan]) ) );

    }
    
    break;
    
  case( iio_op_write ):
    
    /* 
     * Set quadrature encoder counter values. Cater for both positive
     * and negative counter values.
     */
    
    for( chan = first; chan < first + number; ++chan ) {
      
      val = iio_data_get( opnode, chan );

      /* Extract principal and extended domains from <val> */
      
      pd_val = ((val % QUAD_COUNT_MAX) + QUAD_COUNT_MAX ) % QUAD_COUNT_MAX;
      state->ext_counter[chan] = val - pd_val;
      

      /* Update new encoder counter in cache */

      state->enc_counter[chan] = pd_val;

      /*
       * Load the preset/compare register in three bytes, LSB first.
       */

      *reg->enc[chan].cr  = CR_WRITE;             /* Ready PCR register and clear borrow/carry */      
      
      byte = (char)(pd_val & 0xff);               /* LSB */
      *reg->enc[chan].pcr  = byte;                /* Preset/compare register */      
      byte = (char)((pd_val >> 8) & 0xff); 
      *reg->enc[chan].pcr  = byte;                
      byte = (char)((pd_val >> 16) & 0xff);       /* MSB */
      *reg->enc[chan].pcr  = byte;   
      
      /* Transfer the PC preset/compare register to the counter register */
      
      *reg->enc[chan].cr = CR_COUNTER_SET;

      /* Update state channel cache to correspond to new value */
      

    }
    break;
  
    
  default:
    return iio_error("Operation code not supported by channel");
  }

  return iio_status_ok;

}



/*
 * IP-Quadrature initialisation function
 *
 * Initialise the module using the configuration already loaded in the
 * state and register structures. There should be no need to probe, as
 * the IP carrier driver has already done it. Just initialise the
 * registers to match the state (or vice-versa).
 *
 * NB: Hardware module probing (using iio_probe) is not required for
 * IP-modules housed in IP carriers.
 *
 */

HIDDEN IIO_STATUS iio_ipquadrature_init(IIO_MREG *reg, IIO_MSTATE *state) 
{
  
  int chan;

  /* 
   * Verify the module via the manufacturer ID and model number from
   * the IP module PROM.
   */

  uint8_t manufacturer_id = MANUFACTURER_ID;
  uint8_t model_number = MODEL_NUMBER;
  
  iio_eret( iio_ipinfo_ident(reg->slot, manufacturer_id, model_number ) );

  /* 
   * Reset the IP-Quadrature. Set IP module into "standard" quadrature mode.
   */

  for( chan=0; chan < QUAD_CHAN; chan++ ) {
    
    *reg->enc[chan].cr   = CR_RESET;              /* Master reset */
    *reg->enc[chan].icr  = ICR_RESET;             /* Input control register */    
    *reg->enc[chan].qr   = reg->prescaler;        /* Set quadrature prescaler */
    *reg->enc[chan].ocr  = OCR_SET;               /* Output control register */
    *reg->enc[chan].chcr = CHCR_RESET;            /* Channel configuration register */    
    *reg->enc[chan].cr   = CR_BCS_CLEAR;          /* Counter control register */         
  }
#ifdef RS442
  *reg->csr             = CSR_RESET_RS422;         /* Control and status register for RS422 input */
#else
  *reg->csr             = CSR_RESET_TTL;           /* Control and status register for TTL input */
#endif
  
  /*
   * Reset extended domain counters in state structure
   */
  
  for( chan=0; chan < QUAD_CHAN; chan++ ) {
    state->enc_counter[chan] = 0;    
    state->ext_counter[chan] = 0;
  }
  
  return iio_status_ok;
}




/*
 * IP Quadrature installation function
 */

HIDDEN IIO_STATUS iio_ipquadrature_install(IIO_MODULE *module, char *argv[]) {
  
  /*
   * Decode the module driver arguments, resolve the VME base address to
   * logical addresses, build the register structure containing pointers 
   * to the device, and register the channels it provides.
   */
  
  IIO_CHNODE *chnode;
  IIO_MREG *reg;
  int chan;
  int prescaler;

  /* get a zeroed register structure */
  iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
  
  /* get a state structure */
  iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );
  
  /*
   * Get module specific configuration details. Slot and prescaler options are mandatory.
   */
  
  iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );
  iio_eret( iio_arg(argv, "prescaler", iio_arg_int8, &prescaler) );

  /* Check /etc/iio.conf command line parameters ranges */
  
  if (!reg->slot)
    return iio_error( "No slot specified" );

  if( prescaler == 1 )
    reg->prescaler = QR_COUNT_X1;
  else if( prescaler == 2 )
    reg->prescaler = QR_COUNT_X2;
  else if( prescaler == 4 )
    reg->prescaler = QR_COUNT_X4;
  else
    return iio_error( "Invalid prescaler option: Must be 1, 2, or 4." );
  

  /* 
   * Resolve channel specific register addresses. Uses VME bus addressing.
   */


  for( chan = 0;  chan < QUAD_CHAN; chan++ ) {
    iio_eret( iio_resolve_list(
			       reg->slot, iio_space_io,
			       iio_size_8, 0x01 + chan * 4, &reg->enc[chan].ol,
			       iio_size_8, 0x01 + chan * 4, &reg->enc[chan].pcr,
			       iio_size_8, 0x03 + chan * 4, &reg->enc[chan].sr,
			       iio_size_8, 0x03 + chan * 4, &reg->enc[chan].cr,
			       iio_size_8, 0x03 + chan * 4, &reg->enc[chan].icr,
			       iio_size_8, 0x03 + chan * 4, &reg->enc[chan].ocr,
			       iio_size_8, 0x03 + chan * 4, &reg->enc[chan].qr,
			       iio_size_8, 0x21 + chan * 4, &reg->enc[chan].chcr,
			       0
			       )
	      );


  }
  
  /*
   * Common channel registers. Uses VME bus addressing.
   */

  iio_eret( iio_resolve_list(
			     reg->slot, iio_space_io,
			     iio_size_8, 0x31, &reg->csr,
			     iio_size_8, 0x35, &reg->vr,
			     0
			     )
	    );
  
  
  /* 
   * Register the quadrature channels with the channel list.  Note
   * that the 24-bit hardware counter has been software extended to
   * 32-bits...
   */
  
  
  iio_eret(
	   iio_chnode(
		      module,
		      iio_chtype_enc,  
		      32,                                /* 24-bit hardware, but 32-bit in software */
		      QUAD_CHAN, 
		      iio_ipquadrature_op,                     /* operate function for these channels */
		      &chnode
		      )
	   );
    

  for (chan = 0; chan < QUAD_CHAN; ++chan)
    iio_eret(
	     iio_chnode_linear(
			       chnode, chan,
			       1,                       /* scale  */
			       0,                       /* offset */ 
			       " "                      /* Leave the SI units blank... */
  

			       )
	     );
  

  return iio_status_ok;
}



/*
 * IP Quadrature identification function
 *
 * Call iio_minfo to register this module with IIO
 *
 */

IIO_STATUS iio_ipquadrature( void ) {
  return iio_minfo(
		   "ipquad",
		   "GreenSpring IP-Quadrature",
		   "$Revision: 2222 $",
		   iio_multi_yes,                   /* possibly more than one IP installed */
		   iio_ipquadrature_install,
		   iio_ipquadrature_init
		   );
}

