/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipinfo.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipinfo.c -- IndustryPack specific functions
 * Robin Kirkham, July 1996
 */
#include "../internal.h"


IIO_STATUS iio_ipinfo_read(IIO slot, IIO_IPINFO *ipinfo) {
    /*
     * This reads the IndustryPack ID PROM of the IP at the given slot
     * into a USER SUPPLIED IIO_IPACINFO structure. This function is only
     * called by IP module drivers in the init routine.
     *
     * Reading the PROM is a messy procedure, because of the variable
     * inter-byte gaps the address mapping of the IP carrier/bus might
     * introduce. Because of this, each word address has to be separately
     * resolved and read into a buffer.
     *
     * For the convenience of module drivers, the 
     */
    int count;
    volatile uint16_t *prom;

    if (! ipinfo)
	return iio_fatal("NULL IPAC info structure pointer");

    /* resolve first address of prom and probe it */
    iio_eret( iio_resolve(slot, iio_space_id, iio_size_16, 0, (void **)&prom) );
    iio_eret( iio_probe(prom, iio_size_16, iio_ptype_read) );

    /* read the 0x20 bytes into the prom image part of the given structure */
    for (count = 0; count < 0x20; ++count) {
	iio_eret(
	    iio_resolve(
		slot, iio_space_id, iio_size_16,
		count * 2, (void **)&prom
	    )
	);
	ipinfo->prom[count] = *prom;
    }

    /*
     * There are two possible formats for the ID PROM. The format can be
     * inferred from the first couple of words, which are supposed to form
     * human-readable string. It's also possible we find 0xffff there; this
     * means we read an empty socket through a carrier/bus with no bus timeout
     * capability (such as the ISA bus)
     */
    if (
	((ipinfo->prom[0] & 0x00ff) == (uint16_t)'I') &&
	((ipinfo->prom[1] & 0x00ff) == (uint16_t)'P') &&
	((ipinfo->prom[2] & 0x00ff) == (uint16_t)'A') && (
	    ((ipinfo->prom[3] & 0x00ff) == (uint16_t)'C') ||
	    ((ipinfo->prom[3] & 0x00ff) == (uint16_t)'D')
	)
    ) {
	/*
	 * TYPE I ID PROM. This is the original 8-bit GreenSpring format,
	 * defined in the IndustryPack Logic Interface Specification 0.7.1.
	 * The data starts with the string "IPAC" or "IPAD" (for 32-bit IPs).
	 */

	/* clear rubbish out of top bytes of the 16-bit words we read */
	for (count = 0; count < 0x20; ++count)
	    ipinfo->prom[count] &= 0x00ff;

	/* we should check the CRC here. How is this done? */

	/* duplicate manufacturer, product... codes */
	ipinfo->mid = ipinfo->prom[4];
	ipinfo->pid = ipinfo->prom[5];
	ipinfo->rev = ipinfo->prom[6];
	ipinfo->did = ipinfo->prom[8] | (ipinfo->prom[9] << 8);
	ipinfo->flg = 0;

	/* set up pointers to the pack- and user-specific prom */
	ipinfo->pprom = ipinfo->prom + 12;
	ipinfo->uprom = ipinfo->prom + ipinfo->prom[10]/2;

	/* set magic number */
	ipinfo->magic = iio_magic_ipinfo;
	return iio_status_ok;
    }

    if (
	(ipinfo->prom[0] == (((uint16_t)'V' << 8) | (uint16_t)'I')) &&
	(ipinfo->prom[1] == (((uint16_t)'T' << 8) | (uint16_t)'A')) &&
	(ipinfo->prom[2] == (((uint16_t)'4' << 8) | (uint16_t)' '))
    ) {
	/*
	 * TYPE II ID PROM. This is the revved-up 16-bit version from the
	 * IP Module Specification draft 1.0.d.0 from the VITA organisation.
	 * I haven't tested this because I don't have a type II IP. The format
	 * is in draft form as well, so I suppose it might change
	 */

	/* we should check the CRC here. How is this done? */

	/* duplicate manufacturer, product... codes */
	ipinfo->mid = ipinfo->prom[4] | ((ipinfo->prom[3] & 0xff) << 16);
	ipinfo->pid = ipinfo->prom[5];
	ipinfo->rev = ipinfo->prom[6];
	ipinfo->did = ipinfo->prom[8] | (ipinfo->prom[9] << 16);
	ipinfo->flg = ipinfo->prom[10];

	/* set up pointers to the pack- and user-specific prom */
	ipinfo->pprom = ipinfo->prom + 13;
	ipinfo->uprom = ipinfo->prom + ipinfo->prom[11]/2;

	/* set magic number */
	ipinfo->magic = iio_magic_ipinfo;
	return iio_status_ok;
    }


    /* try and give an vaguely informative error */
    if (
	(ipinfo->prom[0] == 0xffff) &&
	(ipinfo->prom[1] == 0xffff)
    ) 
	return iio_error("IP slot is probably empty");

    return iio_error("ID PROM has unrecognised format");
}


HIDDEN IIO_STATUS iio_ipinfo_dump(char *name, IIO_IPINFO *ipinfo) {
    /*
     * Log the contents of an IIO_IPINFO structure
     */
    int count;

    iio_log(
	"%s: mid 0x%x, pid 0x%x, rev 0x%x, did 0x%x, flg 0x%x\n",
	name,
	ipinfo->mid,
	ipinfo->pid,
	ipinfo->rev,
	ipinfo->did,
	ipinfo->flg
    );

    for (count = 0; count < 0x20; count += 8)
	iio_log(
	    "    %04x %04x %04x %04x %04x %04x %04x %04x\n",
	    ipinfo->prom[count + 0],
	    ipinfo->prom[count + 1],
	    ipinfo->prom[count + 2],
	    ipinfo->prom[count + 3],
	    ipinfo->prom[count + 4],
	    ipinfo->prom[count + 5],
	    ipinfo->prom[count + 6],
	    ipinfo->prom[count + 7]
	);

    return iio_status_ok;
}


IIO_STATUS iio_ipinfo_ident(IIO slot, uint32_t mid, uint32_t pid) {
    /*
     * Function used by IP module drivers when they don't need to keep
     * the whole IIO_IPINFO structure, but just want to test to see if
     * the mid and pid codes match
     */
    IIO_IPINFO ipinfo;
    IIO_STATUS stat = iio_ipinfo_read(slot, &ipinfo);

    if (stat || mid != ipinfo.mid || pid != ipinfo.pid) {
	iio_ipinfo_dump(slot->name, &ipinfo);
	return stat ? stat : iio_error("Unexpected ID PROM identity");
    }
    return iio_status_ok;
}


IIO_STATUS iio_ipinfo_show(IIO slot) {
    /*
     * Read and print out the ID PROM information
     */
    IIO_IPINFO ipinfo;

    iio_eret( iio_ipinfo_read(slot, &ipinfo) );
    iio_eret( iio_ipinfo_dump(slot->name, &ipinfo) );
    return iio_status_ok;
}
