/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipdualpit.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipdualpit.c -- driver for GreenSpring IP-Dual PI/T 
 *
 * The IP-Dgital 48 and IP-Dual PI/T provide two Motorola MC68230 PI/T chips.
 * The only practical difference between the two is which PI/T pins are 
 * brought out to the 50-way connector; for the Digital-48 the Port C/dual
 * function pins are available (and are intended for use as digital inputs 
 * or outputs), but for the Dual-PI/T the digital IO handshaking lines are
 * provided. The software implication is that some of the chip modes are made
 * unavailable in certain combinations, as the mode is not useful if certain 
 * pins cannot be connected. 
 *
 * At the moment, configuration of the PI/T chips is FIXED to provide ONLY
 * one 32-bit digital input port. This ignores most all the capability
 * of the MC68230, which is a complicated multi-modal affair. Because we have
 * to use the handshaking pins, we cannot use port C of either chip. Because
 * of that
 *
 * Module options:
 *
 *	-slot <channel>			IP slot
 *	-ipdigital48			the IP is really a IP-Digital 48,
 *					re-jumpered as a IP Dual PI/T
 *
 * I must recommend against using this IP for just about anything. It is 
 * difficult to understand and use, has poor output drive capability, and in
 * any case you don't really get 48 digital input/outputs in mosr of the
 * modes.
 */
#include "../internal.h"
#include "../chip/mc68230.h"


    /* register pointer structure */
struct IIO_MREG {
    IIO slot;				/* our IP slot */
    IIO_MC68230 pit[2];			/* the two PI/T chips */
    IIO_BOOL d48;			/* true if we are a IP-Digital 48 */
};



HIDDEN IIO_STATUS iio_ipdualpit_di32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Operate function for one 32-bit input port, which is the
     * concatenation of ports A and B on both PI/Ts
     */
    volatile uint32_t result;

    switch (op) {
    case iio_op_read:
	/*
	 * THIS REQUIRES the handshake H2 of PI/T X to connect to 
	 * handshake H3 of both PI/T X and PI/T Y. First, we assert H2
	 * (used as an output) which latches data into ports A and B.
	 * Then we read them twice (to get the data we just latched
	 * through the double buffering). Then we clear H2 again.
	 */
	*reg->pit[0].pacr = (
	    (0 << IIO_MC68230_PACR_SUBMODE_roll) &
	    IIO_MC68230_PACR_SUBMODE_mask
	) | (
	    (5 << IIO_MC68230_PACR_H2_CTL_roll) &
	    IIO_MC68230_PACR_H2_CTL_mask
	);

	/* delay here for 50us to allow H2 to rise... */

	/* read the data registers and compute result */
	result =
	    ((uint32_t)*reg->pit[0].padr <<  0) |
	    ((uint32_t)*reg->pit[0].pbdr <<  8) |
	    ((uint32_t)*reg->pit[1].padr << 16) |
	    ((uint32_t)*reg->pit[1].pbdr << 24);

	/* do it again (double buffering) */
	result =
	    ((uint32_t)*reg->pit[0].padr <<  0) |
	    ((uint32_t)*reg->pit[0].pbdr <<  8) |
	    ((uint32_t)*reg->pit[1].padr << 16) |
	    ((uint32_t)*reg->pit[1].pbdr << 24);

	/* release H2 */
	*reg->pit[0].pacr = (
	    (0 << IIO_MC68230_PACR_SUBMODE_roll) &
	    IIO_MC68230_PACR_SUBMODE_mask
	) | (
	    (4 << IIO_MC68230_PACR_H2_CTL_roll) &
	    IIO_MC68230_PACR_H2_CTL_mask
	);

	/* return result */
	iio_eret( iio_data_set(opnode, first, result) );
	break;

    case iio_op_show:
	iio_log("PI/T X:\n");
	iio_mc68230_show(&reg->pit[0]);
	iio_log("PI/T Y:\n");
	iio_mc68230_show(&reg->pit[1]);
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipdualpit_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Check the identity of the IP, then call the functions to "reset"
     * the two PI/T chips. Then set the registers to select the mode we
     * want
     */
    int chip;

    /*
     * Check my identity. If the hardware was originally a IP-Digital 48
     * the MID code should be 0x24, otherwise it should be 0x23
     */
    iio_eret(
	iio_ipinfo_ident(
	    reg->slot,
	    0xf0,
	    reg->d48 ? 0x24 : 0x23
	)
    );

    for (chip = 0; chip < 2; ++chip) {

	/* put the chip ports into Mode 1, submode X0, after a reset */
	iio_eret( iio_mc68230_init(&reg->pit[chip]) );

	/* Mode 1 */
	*reg->pit[chip].pgcr =
	    ((1 << IIO_MC68230_PGCR_PMC_roll) & IIO_MC68230_PGCR_PMC_mask);

	/* Submode X0 */
	*reg->pit[chip].pacr = (
		(0 << IIO_MC68230_PACR_SUBMODE_roll) &
		IIO_MC68230_PACR_SUBMODE_mask
	    ) | (
		(4 << IIO_MC68230_PACR_H2_CTL_roll) &
		IIO_MC68230_PACR_H2_CTL_mask
	);
	*reg->pit[chip].pbcr = (
	    (0 << IIO_MC68230_PBCR_SUBMODE_roll) &
	    IIO_MC68230_PBCR_SUBMODE_mask
	);

	/* set data direction regs for inputs */
	*reg->pit[chip].padr = 0x00;
	*reg->pit[chip].pbdr = 0x00;

	/* no interrrupts please */
	*reg->pit[chip].psrr = 0x00;

	/* Mode 1, enable H32 and H12, make H2 positive true */
	*reg->pit[chip].pgcr =
	    ((1 << IIO_MC68230_PGCR_PMC_roll) & IIO_MC68230_PGCR_PMC_mask) |
	    IIO_MC68230_PGCR_H34_ENABLE |
	    IIO_MC68230_PGCR_H12_ENABLE |
	    IIO_MC68230_PGCR_H2_SENSE;

	/* iio_mc68230_show(&reg->pit[chip]); */
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipdualpit_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the IP-specific arguments, create a register structure, and
     * the then call the chip driver to install the channels for each of
     * the two chips on our behalf
     */
    IIO_MREG *reg;
    unsigned chip;

    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get the slot argument */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );
    if (!reg->slot)
	return iio_error("No IP slot specified");

    /* see if we are a re-jumpered IP-Digital 48 */
    iio_eret( iio_arg(argv, "ipdigital48", iio_arg_bool, &reg->d48) );


    /* install the PI/T chips (initialise register structure) */
    for (chip = 0; chip < 2; ++chip) {
	iio_eret(
	    iio_mc68230_install(
		&reg->pit[chip],
		reg->slot,
		iio_space_io,
		(chip ? 0x41 : 0x1), 1
	    )
	);
    }

    /* register the one di32 channel */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_di, 32, 1,
	    iio_ipdualpit_di32,
	    NULL
	)
    );

    return iio_status_ok;
}


IIO_STATUS iio_ipdualpit(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "ipdualpit",
	    "GreenSpring IP-Dual PI/T",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_ipdualpit_install,
	    iio_ipdualpit_init
	)
    );
    return iio_status_ok;
}
