/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipdigital24.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipdigital24.c -- driver for GreenSpring IP-Digital 24
 *
 * The IP-Digital-24 provides 24 digital IO lines, organised as three
 * banks of 8-bit registers. The input and open-collector output channels
 * are linked, making them OCIO channels.
 *
 * Module options:
 *
 *	-slot <channel>			IP slot
 */
#include "../internal.h"


    /* register pointer structure */
struct IIO_MREG {
    IIO slot;				/* our IP slot */
    volatile uint8_t *odr[3];	/* output register pointers */
    volatile uint8_t *idr[3];	/* input register pointers */
};


HIDDEN IIO_STATUS iio_ipdigital24_ocio(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * We regard the device as three byte-wide registers. This isn't right
     * as we could access one as 16-but and one as 8, but I haven't time
     * at the moment.
     *
     * Note the output data is INVERTED with respect to the IIO convention
     * i.e., we have to invert it
     */
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->odr[seqno] = ~(uint8_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, ~*reg->odr[seqno] & 0xff) );
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->idr[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipdigital24_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Since the device has readback registers, there is no need for
     * a state structure. Hoever, we clear the output registers anyway
     */

    /* check my identity */
    iio_eret( iio_ipinfo_ident(reg->slot, 0xf0, 0x11) );

    /* clear the lot */
    *reg->odr[0] = 0xff;
    *reg->odr[1] = 0xff;
    *reg->odr[2] = 0xff;
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipdigital24_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;

    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get the slot argument */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );

    /* this board always uses D16 short VME addressing */
    if (!reg->slot)
	return iio_error("No IP slot specified");

    /* resolve local register addresses to virtual */
    iio_eret(
	iio_resolve_list(
	    reg->slot, iio_space_io,
	    iio_size_8, 0x1, &reg->odr[0],
	    iio_size_8, 0x0, &reg->odr[1],
	    iio_size_8, 0x3, &reg->odr[2],
	    iio_size_8, 0x5, &reg->idr[0],
	    iio_size_8, 0x4, &reg->idr[1],
	    iio_size_8, 0x7, &reg->idr[2],
	    0
	)
    );

    /* register the channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_ocio, 8, 3,
	    iio_ipdigital24_ocio, NULL
	    )
	);

    return iio_status_ok;
}


IIO_STATUS iio_ipdigital24(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"ipdigital24",
	"GreenSpring IP-Digital 24",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_ipdigital24_install,
	iio_ipdigital24_init
    );
}
