/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: ipdac.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * ipdac.c -- driver for GreenSpring IP-DAC
 *
 * The IP-DAC provides six 12-bit DAC outputs, with output ranges selected
 * by on-board jumpers.
 *
 * Module options:
 *
 *	-slot <channel>			IP slot
 *	-range <range>			Specify output range for all DACs
 *	-range.0 <range>		Specify output range for DAC0
 *	-range.1 <range>		Specify output range for DAC1
 *	-range.2 <range>		Specify output range for DAC2
 *	-range.3 <range>		Specify output range for DAC3
 *	-range.4 <range>		Specify output range for DAC4
 *	-range.5 <range>		Specify output range for DAC5
 *
 * <range> is a number:
 *	0	0 to +5V
 *	1	0 to +10V
 *	2	-2.5 to +2.5V
 *	3	-5 to +5V
 *	4	-10 to +10V
 *
 * The correction factors in the IDPROM are not yet used. The current loop
 * drive option for channels 0 and 1 is not implemented.
 */
#include "../internal.h"


    /* register pointer structure */
struct IIO_MREG {
    IIO slot;				/* our IP slot */
    volatile uint16_t *dac[6];	/* DAC output register pointers */
    int ioffset[6];			/* integer offset value */
    IIO_BOOL cloop;			/* current loop factory option */
};

    /* state data structure */
struct IIO_MSTATE {
    volatile uint16_t dac[6];	/* data register readback */
};

    /* output range table */
#define NRANGES 5
struct {
    int ioffset;
    double scale;
    char *name;
} iio_ipdac_range[NRANGES] = {
    { 0x0,	5.00/4096,	"0 to +5V" },
    { 0x0,	10.0/4096,	"0 to +10V" },
    { 0x800,	5.00/4096,	"-2.5 to +2.5V" },
    { 0x800,	10.0/4096,	"-5 to +5V" },
    { 0x800,	20.0/4096,	"-10 to +10V" },
};

    /* channel local address table */
#define NCHAN 6
HIDDEN unsigned iio_ipdac_laddr[NCHAN] = {
    0x4, 0x2, 0xc, 0xa, 0x14, 0x12
};


HIDDEN IIO_STATUS iio_ipdac_dac(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Operate on the DACs. If we write, save the value we actually
     * wrote (but clipped to 12-bits) for the readback. 
     */
    int val, seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    val = iio_data_get(opnode, seqno) + reg->ioffset[seqno];
	    val = (val > 0xfff) ? 0xfff : val;
	    val = (val < 0) ? 0 : val;
	    (*reg->dac[seqno]) = state->dac[seqno] = (uint16_t)val;
	    break;

	case iio_op_readback:
	    iio_eret( 
		iio_data_set(
		    opnode, seqno,
		    (int)state->dac[seqno] - reg->ioffset[seqno]
		)
	    );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipdac_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialise the module using the configuration already loaded in the
     * state and register structures. There should be no need to probe,
     * as the IP carrier driver has already done it. Just initialise the
     * registers to match the state (or vice-versa).
     *
     * Beacuse there is no read-back from the device, we have to clear
     * the outputs and the output shadows
     */
    int count;

    /* check our identity */
    iio_eret( iio_ipinfo_ident(reg->slot, 0xf0, 0x16) );

    /* clear the DAC outputs and shadows */
    for (count = 0; count < NCHAN; ++count) 
	*reg->dac[count] = state->dac[count] = reg->ioffset[count];

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_ipdac_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_CHNODE *chnode;
    IIO_MREG *reg;
    uint8_t crange = 0, range[6];
    int count;

    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* get the slot argument */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );

    /* -range set all the channel ranges */
    iio_eret( iio_arg(argv, "range", iio_arg_uint8, &crange) );
    for (count = 0; count < NCHAN; ++count) {
	range[count] = crange;
	iio_eret(
	    iio_arg_index(
		argv, "range", count, iio_arg_uint8, &range[count]
	    )
	);
    }

    /* CURRENT LOOP OPTION NOT IMPLEMENTED */

    if (!reg->slot)
	return iio_error("No slot specified");

    /* register the channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_dac, 12, NCHAN,
	    iio_ipdac_dac, 
	    &chnode
	)
    );

    for (count = 0; count < NCHAN; ++count) {

	/* check the range option value */
	if (range[count] >= NRANGES)
	    return iio_error("-range option out of range");

	/* resolve local register addresses to vitual */
	iio_eret(
	    iio_resolve(
	        reg->slot,
		iio_space_io, iio_size_16,
		iio_ipdac_laddr[count],
		(void **)&reg->dac[count]
	    )
	);

	/* set the ioffset value for the given range */
	reg->ioffset[count] = iio_ipdac_range[range[count]].ioffset;

	/* set the channel scaling values */
	iio_eret(
	    iio_chnode_linear(
		chnode, count, 
		iio_ipdac_range[range[count]].scale,
		0.0, "V"
	    )
	);
    }

    return iio_status_ok;
}


IIO_STATUS iio_ipdac(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"ipdac",
	"GreenSpring IP-DAC",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_ipdac_install,
	iio_ipdac_init
    );
}
