/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id: dmm32at.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * dmm32at - driver for the Diamond MM-32-AT PC-104 multi-IO card
 *
 *	-bus <channel>			ISA bus (default isa.0)
 *	-address <addr>			bus address in ISA port space (def 300)
 *
 *	-adcconfig <config>		ADC input config code (manual page 9)
 *	-adcrange <range>		ADC range code (page 22)
 *	-adcscan <scan>			ADC scan rate (page 20)
 *
 *	-dacrange <range>		DAC range code 
 *	-dacmax <voltage>		DAC full scale voltage (ranges 4 and 5)
 *
 *	-dioconfig <config>		main digital IO config code 
 *	-no-auxdo 			disable auxiliary digital out
 *	-no-auxdi 			disable auxiliary digital in
 */

#include "../internal.h"

    /* default base address */
#define BASE 0x300

    /* list of allowable base addresses */
#define NBASES 8
uint16_t iio_dmm32at_bases[] = {
    0x100, 0x140, 0x180, 0x200, 0x280, 0x300, 0x340, 0x380
};


    /*
     * Register bit definitions 
     *
     * All of the registers are read-write and many have read functions
     * completely unrelated to their write functions. They are also
     * somewhat jumbled up. This makes things very hard to understand.
     *
     * The w or r in the bit definitions means write or read; neither
     * indicates both write and read.
     *
     * Also the 16-bit ADC and 12-bit DAC data has to be accessed
     * bytewise (!) and through a non-addressable portal
     */

#define ADCL_w_SOC		(1<<0)	/* start conversion */
#define ADCH_r_ADCL_mask	0xff	/* ADC low byte */

#define ADCH_w_DOUT_mask	0x7	/* auxiliary digital out */
#define ADCH_r_ADCH_mask	0xff	/* ADC high byte */

#define DACL_w_mask		0xff	/* DAC low byte */
#define DACL_r_DACBSY		(1<<7)	/* DAC busy */
#define DACL_r_CALBSY		(1<<7)	/* EEPROM calibrate busy */
#define DACL_r_DIN_mask		0xf	/* auxiliary digital in */

#define DACH_w_DACCH_mask	0x3	/* DAC channel mask */
#define DACH_w_DACCH_roll	6	/* DAC channel roll */

#define FIFOC_w_FIFOEN		(1<<3)	/* FIFO enable */
#define FIFOC_w_SCANEN		(1<<2)	/* scan enable */
#define FIFOC_w_FIFORST		(1<<1)	/* FIFO reset */
#define FIFOC_r_EF		(1<<7)	/* empty flag */
#define FIFOC_r_HF		(1<<6)	/* half full flag */
#define FIFOC_r_FF		(1<<5)	/* full flag */
#define FIFOC_r_OVF		(1<<4)	/* overflow flag */
#define FIFOC_r_FIFOEN		(1<<3)	/* FIFO enable */
#define FIFOC_r_SCANEN		(1<<2)	/* scan enable */
#define FIFOC_r_PAGE_8254	0x0	/* 8254 page enabled */
#define FIFOC_r_PAGE_8255	0x1	/* 8255 page enabled */
#define FIFOC_r_PAGE_CAL	0x3	/* calibrate page enabled */

#define MCR_w_RESETA		(1<<5)	/* reset all except counters */
#define MCR_w_RESETD		(1<<4)	/* reset all except counters and DACS */
#define MCR_w_INTRST		(1<<3)	/* reset interrupter */
#define MCR_w_PAGE_8254		0x0	/* 8254 page enable */
#define MCR_w_PAGE_8255		0x1	/* 8255 page enable */
#define MCR_w_PAGE_CAL		0x3	/* calibrate page enable */
#define MCR_w_PAGE_mask		0x3	/* page enable mask */
#define MCR_r_STS		(1<<7)	/* ADC scan in progress */
#define MCR_r_SD_mask		0x3	/* ADC SE/DI mask */
#define MCR_r_SD_roll		5	/* ADC SE/DI roll */
#define MCR_r_ADCH_mask		0x1f	/* next ADC channel */

#define OCR_w_ADINTE		(1<<7)	/* ADC interrupt enable */
#define OCR_w_DINTE		(1<<6)	/* digital interrupt enable */
#define OCR_w_TINTE		(1<<5)	/* timer interrupt enable */
#define OCR_w_CLKEN		(1<<1)	/* ADC sample clock enable */
#define OCR_w_CLKSEL		(1<<0)	/* select internal clock */
#define OCR_r_ADINT		(1<<7)	/* ADC interrupt occurred */
#define OCR_r_DINT		(1<<6)	/* digital interrupt occurred */
#define OCR_r_TINT		(1<<6)	/* timer interrupt occurred */
#define OCR_r_CLKEN		(1<<1)	/* ADC sample clock enable */
#define OCR_r_CLKSEL		(1<<0)	/* select internal clock */

#define CTCR_FREQ12		(1<<7)	/* counters 1 and 2 frequency */
#define CTCR_FREQ0		(1<<6)	/* counter 0 frequency */
#define CTCR_OUT2EN		(1<<5)	/* counter 2 output enable */
#define CTCR_OUT0EN		(1<<4)	/* counter 0 output enable */
#define CTCR_GT0EN		(1<<2)	/* counter 0 gate enable */
#define CTCR_SRC0		(1<<1)	/* counter 0 source */
#define CTCR_GT12EN		(1<<0)	/* counters 1 and 2 gate enable */

#define ACR_w_SCINT_mask	0x3	/* ADC scan interval mask */ 
#define ACR_w_SCINT_roll	4	/* ADC scan interval roll */ 
#define ACR_w_RANGE		(1<<3)	/* input range */
#define ACR_w_ADBU		(1<<2)	/* unipolar */
#define ACR_w_GAIN_mask		0x3	/* ADC gain mask */ 
#define ACR_r_WAIT		(1<<7)	/* ADC settling in progress */ 
#define ACR_r_RANGE		(1<<3)	/* input range */
#define ACR_r_ADBU		(1<<2)	/* unipolar */
#define ACR_r_GAIN_mask		0x3	/* ADC gain mask */ 

#define PR3_8255_SET		(1<<7)	/* set bit, always 1 */
#define PR3_8255_MODEC		(1<<6)	/* port C mode */
#define PR3_8255_MODEA		(1<<5)	/* port A mode */
#define PR3_8255_DIRA		(1<<4)	/* port A input */
#define PR3_8255_DIRCH		(1<<3)	/* port C high input */
#define PR3_8255_MODEB		(1<<2)	/* port B mode */
#define PR3_8255_DIRB		(1<<1)	/* port B input */
#define PR3_8255_DIRCL		(1<<0)	/* port C low input */


    /* register data structure */
struct IIO_MREG {

    /* register port addresses */
    volatile uint8_t
        *adcl,				/* ADC LSB, start convert */
        *adch,				/* ADC MSB, aux digital out */
        *adchl,				/* ADC channel register low */
        *adchh,				/* ADC channel register high */
        *dacl,				/* DAC LSB, aux digital out */
        *dach,				/* DAC MSB and channel, update */
        *fifod,				/* ADC FIFO depth */
        *fifoc,				/* ADC FIFO control */
        *mcr,				/* misc control register */
        *ocr,				/* operation control register */
        *ctcr,				/* counter/timer control register */
        *acr,				/* ADC configuration register */

        *pr0,				/* 8254/8255/calibrate registers */
        *pr1,
        *pr2,
        *pr3;

    /* configuration info */
    int adcioffset, dacioffset;		/* ACD/DAC integer offset values */
    uint8_t adcconfig;		/* ADC input configuration */
    uint8_t acrvalue;		/* value for acr */
    uint8_t dioconfig;		/* mode (0-7) of 8255 */
    IIO_BOOL auxdo, auxdi;		/* flags indicating aux port use */
};

    /* state data structure */
#define NDACCHAN 4
struct IIO_MSTATE {
    uint16_t dac[NDACCHAN];		/* DAC readbacks */
    uint8_t auxdo;			/* shadow for auxdo port */
};

    /* ADC output range table */
#define NADCRANGES 16
struct {
    int ioffset;			/* channel value for 0 volts */
    double scale;			/* volts per bit */
    int valid;				/* 1 if range is allowed */
    char *name;				/* name of range */
} iio_dmm32at_adcrange[NADCRANGES] = {
    { 0x0,	10.0/65536,	1,	"-5 to +5V" },
    { 0x0,	5.0/65536,	1,	"-2.5 to +2.5V" },
    { 0x0,	2.5/65536,	1,	"-1.25 to +1.25V" },
    { 0x0,	1.25/65536,	1,	"-0.625 to +0.625V" },
    { 0x0,	0.0,		0,	"invalid" },
    { 0x0,	0.0,		0,	"invalid" },
    { 0x0,	0.0,		0,	"invalid" },
    { 0x0,	0.0,		0,	"invalid" },
    { 0x0,	20.0/65536,	1,	"-10 to +10V" },
    { 0x0,	10.0/65536,	1,	"-5 to +5V" },
    { 0x0,	5.0/65536,	1,	"-2.5 to +2.5V" },
    { 0x0,	2.5/65536,	1,	"-1.25 to +1.25V" },
    { -32768,	10.0/65536,	1,	"0 to +10V" },
    { -32768,	5.0/65536,	1,	"0 to +5V" },
    { -32768,	2.5/65536,	1,	"0 to +2.5V" },
    { -32768,	1.25/65536,	1,	"0 to +1.25V" }
};

    /*
     * ADC input config table. Input config is set by jumpers, and the
     * selected config has to be given in the -adcconfig argument.
     * They are checked against the value reflected in the MCR.
     * We don't support config 2 at present, as the channels have
     * a gap which complicates the operation function. Mode 3
     * has exactly the same complement of channels anyway
     */
#define NADCCONFIGS 4
struct {
    unsigned nchan;			/* number of channels in the config */
    int valid;				/* 1 if this config allowed */
    unsigned sd;			/* expected value of SD bits in MCR */
    char *name;				/* name of config */
} iio_dmm32at_adcconfig[NADCCONFIGS] = {
    { 32,	1,	3,	"0-31 SE" },
    { 16,	1,	0,	"0-15 DI" },
    { 24,	0,	2,	"0-7 DI, 8-15 SE, 24-31 SE" },
    { 24,	1,	1,	"0-7 SE, 8-15 DI, 16-23 SE" }
};


    /* DAC output range table */
#define NDACRANGES 6
struct {
    int ioffset;			/* channel value for 0 volts */
    double scale;			/* volts per bit */
    int prog;				/* 1 if full scale programmable */
    char *name;				/* name of range */
} iio_dmm32at_dacrange[NDACRANGES] = {
    { 0x800,	10.0/4096,	0,	"-5 to +5V" },
    { 0x0,	5.0/4096,	0,	"0 to +5V" },
    { 0x800,	20.0/4096,	0,	"-10 to +10V" },
    { 0x0,	10.0/4096,	0,	"0 to +10V" },
    { 0x800,	2.0/4096,	1,	"-p to +pV" },
    { 0x0,	1.0/4096,	1,	"0 to +pV" }
};


    /* 8255 digital IO chip mode table (page 34) */
#define NDIOCONFIGS 8
struct {
    unsigned ndi, ndo;			/* number of di and do channels */
    uint8_t pr3;			/* value for PR3 register */
    unsigned int sdi[3];		/* register offsets of di channels */
    unsigned int sdo[3];		/* register offsets of do channels */
} iio_dmm32at_dioconfig[NDIOCONFIGS] = {
    { 3,  0,  0x9b,  { 0, 1, 2 },  { 9, 9, 9 } },
    { 2,  1,  0x92,  { 0, 1, 9 },  { 2, 9, 9 } },
    { 2,  1,  0x99,  { 0, 2, 9 },  { 1, 9, 9 } },
    { 1,  2,  0x90,  { 0, 9, 9 },  { 1, 2, 9 } },
    { 2,  1,  0x8b,  { 1, 2, 9 },  { 0, 9, 9 } },
    { 1,  2,  0x82,  { 1, 9, 9 },  { 0, 2, 9 } },
    { 1,  2,  0x89,  { 2, 9, 9 },  { 0, 1, 9 } },
    { 0,  3,  0x80,  { 9, 9, 9 },  { 0, 1, 2 } }
};


HIDDEN IIO_STATUS iio_dmm32at_adc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * ADC operation function
     * Ugh
     *
     */
    unsigned seqno;
    int16_t value;

    /* we do only read operations */
    if (op != iio_op_read)
        return iio_error("Operation code not supported by channel");

    /* set up the channel scan registers */
    iio_port_set8(reg->adchl, first);
    iio_port_set8(reg->adchh, first + number - 1);

    /* for each channel in the range */
    for (seqno = first; seqno < first + number; ++seqno) {

	/* channel gain change would go here, I suppose */
	iio_port_set8(reg->acr, reg->acrvalue);

	/* wait for channel/gain change to complete */
	while (iio_port_get8(reg->acr) & ACR_r_WAIT) ;

	/* start conversion sequence */
	iio_port_set8(reg->adcl, 0);

	/* wait for conversion to complete */
	while (iio_port_get8(reg->mcr) & MCR_r_STS) ;

	/* read and assemble result */
	value =
	    (iio_port_get8(reg->adcl) << 0) |
	    (iio_port_get8(reg->adch) << 8);

        /* return result */
        iio_eret( iio_data_set(opnode, seqno, (int)value - reg->adcioffset) );
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_dac(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * DAC operation function. The four DACs are loaded byte-by-byte,
     * and then a promulgate register has to be written after each.
     *
     * Note that the sequence that appears to work is *different* from
     * that in the manual
     */
    unsigned seqno;
    int val;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:

	    /* compute and limit the DAC value */
	    val = (int)iio_data_get(opnode, seqno) + reg->dacioffset;
	    val = (val > 0xfff) ? 0xfff : val;
	    val = (val < 0) ? 0 : val;
	    
	    /* save value for readback */
	    state->dac[seqno] = (uint16_t)val;

	    /* compute low and high bytes, latter is munged with channel no */
	    iio_port_set8(reg->dacl, val & 0xff);
	    iio_port_set8(reg->dach, (val >> 8) | (seqno << 6));

	    /* poll until DAC ready goes low */
	    while (iio_port_get8(reg->dacl) & DACL_r_DACBSY) ;

	    /* read port initiates DAC update */ 
	    (void)iio_port_get8(reg->dach);

	    break;

	case iio_op_readback:
	    iio_eret(
		iio_data_set(
		    opnode, seqno,
		    (int)state->dac[seqno] - reg->dacioffset
		)
	    );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_auxdi(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Auxiliary digital input, a four-bit afterthought on the main
     * connector. This is normally enabled. If disabled, the pis are
     * used as counter anc ADC conversion inputs
     */
    if (op == iio_op_read) {
        iio_data_set(opnode, first, iio_port_get8(reg->dacl) & DACL_r_DIN_mask);
    } else
        return iio_error("Operation code not supported by channel");

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_auxdo(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Auxiliary digital output, a three-bit afterthought on the main
     * connector. This is normally enabled. If disabled, the pins are
     * outputs from the counter-timer chip
     */

    switch (op) { 
    case iio_op_readback:
	iio_data_set(opnode, first, state->auxdo);
	break;

    case iio_op_write:
	state->auxdo = iio_data_get(opnode, first) & ADCH_w_DOUT_mask;
	iio_port_set8(reg->adch, state->auxdo);
	break;

    case iio_op_clear:
	state->auxdo = 0;
	iio_port_set8(reg->adch, state->auxdo);
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * 8255 digital input ports operation function. There may be between
     * 0 and 3 8-bit input port, depending on the configuration. The
     * di[] array in iio_dmm32at_dioconfig[] maps seqno to register for the
     * given config
     */
    unsigned seqno;

    /* select the 8255 */
    /* iio_port_set8(reg->mcr, MCR_w_PAGE_8255); */

    if (op == iio_op_read) {
	for (seqno = first; seqno < first + number; ++seqno) {

	    /* consistency check */
	    if (iio_dmm32at_dioconfig[reg->dioconfig].sdi[seqno] == 9)
		iio_fatal("8255 seqno map consistency check");

	    iio_data_set(
		opnode,
		first,
		iio_port_get8(
		    reg->pr0 + iio_dmm32at_dioconfig[reg->dioconfig].sdi[seqno]
		)
	    );
	}
    } else
        return iio_error("Operation code not supported by channel");

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_do(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * 8255 digital output ports operation function. There may be between
     * 0 and 3 8-bit output port, depending on the configured mode.
     * do[] array in iio_dmm32at_dioconfig[] maps seqno to register for the
     * given mode
     */
    unsigned seqno;

    /* select the 8255 */
    /* iio_port_set8(reg->mcr, MCR_w_PAGE_8255); */

    for (seqno = first; seqno < first + number; ++seqno) {

	/* consistency check */
	if (iio_dmm32at_dioconfig[reg->dioconfig].sdo[seqno] == 9)
	    iio_fatal("8255 seqno map consistency check");

	switch (op) {
	case iio_op_write:
	    iio_port_set8(
		reg->pr0 + iio_dmm32at_dioconfig[reg->dioconfig].sdo[seqno],
		iio_data_get(opnode, seqno)
	    );
	    break;

	case iio_op_readback:
	    /* use on-chip readback */
	    iio_data_set(
		opnode,
		first,
		iio_port_get8(
		    reg->pr0 + iio_dmm32at_dioconfig[reg->dioconfig].sdo[seqno]
		)
	    );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Board initialisation. Start with a board reset and go from there.
     * It's a complicated board
     */
    unsigned seqno;

    /* reset board, disable interrupts */
    iio_port_set8(reg->mcr, MCR_w_RESETA);
    iio_port_set8(reg->ocr, 0);

    /* set the ADC gain value computed previously */
    iio_port_set8(reg->acr, reg->acrvalue);

    /* wait for gain change to complete */
    while (iio_port_get8(reg->acr) & ACR_r_WAIT) ;

    /* check ADC input mode was specified correctly */
    if (
	((iio_port_get8(reg->mcr) >> MCR_r_SD_roll) & MCR_r_SD_mask) !=
	iio_dmm32at_adcconfig[reg->adcconfig].sd
    )
	return iio_error("ADC input mode jumpers don't match -adcconfig");

    /* clear DACs and shadows */
    for (seqno = 0; seqno < NDACCHAN; ++seqno) {

	/* initialise value for readback */
	state->dac[seqno] = (uint16_t)reg->dacioffset;

	/* compute low and high bytes, latter is munged with channel no */
	iio_port_set8(reg->dacl, reg->dacioffset & 0xff);
	iio_port_set8(reg->dach, (reg->dacioffset >> 8) | (seqno << 6));

	/* poll until DAC serial trasfer finishes  */
	while (iio_port_get8(reg->dacl) & DACL_r_DACBSY) ;

	/* read port initiates DAC update */ 
	(void)iio_port_get8(reg->dach);
    }

    /* set up auxiliary digital bits, disable counter */
    iio_port_set8(reg->ctcr, 0);
    iio_port_set8(reg->adch, state->auxdo = 0);

    /* set up 8255 digital chip (hardware clears outputs) */
    iio_port_set8(reg->mcr, MCR_w_PAGE_8255);
    iio_port_set8(reg->pr3, iio_dmm32at_dioconfig[reg->dioconfig].pr3);

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_dmm32at_install(IIO_MODULE *module, char *argv[]) {
    /*
     * This board is really complicated and hard to understand, let alone
     * configure. We simplify things a bit by leaving out the counter/timer,
     * so the auxiliary IO bits stay as IO. In some cases we leave mode
     * indicatirs in the register structure so the init and operation
     * functions know what's going on, and in other we work out a register
     * value that the init function can use directly
     */
    IIO chan = NULL;
    IIO_MREG *reg;
    IIO_CHNODE *chnode;
    uint16_t base = BASE;
    uint8_t range, scan = 0;
    double scale = 1.0;
    int count, flag = 0;

    /* allocate register and state structures */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* this module always uses the ISA Bus */
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &chan) );
    if (! chan)
	iio_eret( iio_open("isa.0", 0, &chan) );

    /* decode and check module address */
    iio_eret( iio_arg(argv, "address", iio_arg_uint16, &base) );
    for (count = 0; count < NBASES; ++count)
	if (iio_dmm32at_bases[count] == base)
	    ++flag;
    if (! flag)
	return iio_error("Bad module address");

    /* resolve and map registers */
    iio_eret( iio_map(chan, iio_space_port, base, 16) );
    iio_eret(
	iio_resolve_list(
	    chan, iio_space_port,
	    iio_size_8, base + 0x0, &reg->adcl,
	    iio_size_8, base + 0x1, &reg->adch,
	    iio_size_8, base + 0x2, &reg->adchl,
	    iio_size_8, base + 0x3, &reg->adchh,
	    iio_size_8, base + 0x4, &reg->dacl,
	    iio_size_8, base + 0x5, &reg->dach,
	    iio_size_8, base + 0x6, &reg->fifod,
	    iio_size_8, base + 0x7, &reg->fifoc,
	    iio_size_8, base + 0x8, &reg->mcr,
	    iio_size_8, base + 0x9, &reg->ocr,
	    iio_size_8, base + 0xa, &reg->ctcr,
	    iio_size_8, base + 0xb, &reg->acr,
	    iio_size_8, base + 0xc, &reg->pr0,
	    iio_size_8, base + 0xd, &reg->pr1,
	    iio_size_8, base + 0xe, &reg->pr2,
	    iio_size_8, base + 0xf, &reg->pr3,
	    0
	)
    );
    iio_close(chan);


    /* ----------------------------------------------------------- ADC SETUP */

    /* ADC mode determines number of channels */
    reg->adcconfig = 0;
    iio_eret( iio_arg(argv, "adcconfig", iio_arg_uint8, &reg->adcconfig) );
    if (
	(reg->adcconfig >= NADCCONFIGS) ||
	!iio_dmm32at_adcconfig[reg->adcconfig].valid
    )
	return iio_error("Bad -adcconfig argument");

    /* register ADC channels */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 16, 
	    iio_dmm32at_adcconfig[reg->adcconfig].nchan,
	    iio_dmm32at_adc, &chnode
	)
    );

    /* ADC range gives channel scale factors */
    range = 0;
    iio_eret( iio_arg(argv, "adcrange", iio_arg_uint8, &range) );
    if ((range >= NADCRANGES) || (! iio_dmm32at_adcrange[range].valid))
	return iio_error("Bad -adcrange argument");

    reg->adcioffset = iio_dmm32at_adcrange[range].ioffset;
    scale = iio_dmm32at_adcrange[range].scale;
    for (
	count = 0;
	count < iio_dmm32at_adcconfig[reg->adcconfig].nchan;
	++count
    ) {
	iio_eret( iio_chnode_linear(chnode, count, scale, 0.0, "V") );
    }

    /* ADC scan settling time */
    /* DEFAULT SCAN SETTING IS 1 NOT 2 */
    /* scan rates 2 and 3 are broken (sets gain to 1!) */
    scan = 1;
    iio_eret( iio_arg(argv, "adcscan", iio_arg_uint8, &scan) );
    if (scan >= 4)
	return iio_error("Bad -adcscan argument");

    /* work out value for the ACR; range encodes several bits */
    reg->acrvalue = range | (scan << ACR_w_SCINT_roll);


    /* ----------------------------------------------------------- DAC SETUP */

    /* register the four DAC channels */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_dac, 12, NDACCHAN,
	    iio_dmm32at_dac, &chnode
	)
    );

    /* DAC range is set by jumpers and must match arguments */
    range = 0;
    iio_eret( iio_arg(argv, "dacrange", iio_arg_uint8, &range) );
    if (range >= NDACRANGES)
	return iio_error("Bad -dacrange argument");

    /* compute DAC scaling factor */
    if (iio_dmm32at_dacrange[range].prog) {

	/* need DAC programmed output voltage */
	scale = 1000.0;
	iio_eret( iio_arg(argv, "dacmax", iio_arg_double, &scale) );
	if (scale > 999.0)
	    return iio_error("A -dacmax argument is needed for this -dacrange");
	if ((scale < 0.0) || (scale > 10.0))
	    return iio_error("Bad -dacmax argument");

	/* turn it into an iio scale factor */
	scale *= iio_dmm32at_dacrange[range].scale;
    } else
	scale = iio_dmm32at_dacrange[range].scale;

    /* set DAC scaling factors */
    reg->dacioffset = iio_dmm32at_dacrange[range].ioffset;
    for (count = 0; count < NDACCHAN; ++count) {
	iio_eret( iio_chnode_linear(chnode, count, scale, 0.0, "V") );
    }


    /* --------------------------------------------------- AUX DIGITAL SETUP */

    /* the four-bit auxiliary digital in port */
    reg->auxdi = 1;
    iio_eret( iio_arg(argv, "auxdi", iio_arg_bool, &reg->auxdi) );
    if (reg->auxdi) {
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_di, 4, 1,
		iio_dmm32at_auxdi, &chnode
	    )
	);
    }

    /* the three-bit auxiliary digital out port */
    reg->auxdo = 1;
    iio_eret( iio_arg(argv, "auxdo", iio_arg_bool, &reg->auxdo) );
    if (reg->auxdo) {
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_do, 3, 1,
		iio_dmm32at_auxdo, &chnode
	    )
	);
    }


    /* -------------------------------------------------- 8254 DIGITAL SETUP
     *
     * We support the on-board 8255 DIO chip in a limited way with
     * "configurations" which determine how many and which ports are inputs
     * and outputs. A table in iio_dmm32at_dioconfig[] holds the numbers
     * and seqno->reg maps for each config. We ignore the port C splitting
     * and only use Mode 0 of the chip
     */

    reg->dioconfig = 0;
    iio_eret( iio_arg(argv, "dioconfig", iio_arg_uint8, &reg->dioconfig) );
    if (reg->dioconfig >= NDIOCONFIGS)
	return iio_error("Bad -dioconfig argument");

    /* register di channels, if any */
    if (iio_dmm32at_dioconfig[reg->dioconfig].ndi) {
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_di, 8,
		iio_dmm32at_dioconfig[reg->dioconfig].ndi,
		iio_dmm32at_di, &chnode
	    )
	);
    }

    /* register do channels, if any */
    if (iio_dmm32at_dioconfig[reg->dioconfig].ndo) {
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_do, 8,
		iio_dmm32at_dioconfig[reg->dioconfig].ndo,
		iio_dmm32at_do, &chnode
	    )
	);
    }

    /* ---------------------------------------------------- 8255 TIMER SETUP */

    return iio_status_ok;
}


IIO_STATUS iio_dmm32at(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"dmm32at",
	"Diamond MM-32-AT multi-IO",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_dmm32at_install,
	iio_dmm32at_init
    );
}
