/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: bvmipadc.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * bvmipadc.c -- driver for BVM IP-ADC
 *
 * The BVM IP-ADC (different to the GreenSpring IP-ADC) provides sixteen
 * isolated, differential 12-bit ADCs. A built-in timer covering the channel
 * select settling period simplifies the driver considerably. Input is 0-5V
 * or 0-20mA if the 250 ohm current loop sense resistors are added.
 *
 * Module options:
 *
 *	-slot <channel>			IP channel
 *	-gain <gain>			preamplifier gain (computed from R1)
 *	-cloop/-no-cloop		current loop option
 *	-poll/-no-poll			polling read instead of read-wait
 *
 * The ADC correction factors stored in the IDPROM are not as yet used.
 * Otherwise this driver is complete.
 */
#include "../internal.h"

#define NCHAN 16

    /* register pointer structure */
struct IIO_MREG {
    IIO slot;				/* the IP slot we are in */

    volatile uint16_t
	*chselcnv,			/* channel select and start conv */
	*adcst,				/* ADC data immediate */ 
	*chcnv,				/* start conversion */
	*adcimm,			/* ACD data delayed */
	*intvec,			/* interrupt vector */
	*intstat,			/* interrupt status */
	*promcntl;			/* PROM write control */

    IIO_BOOL poll;			/* spinlock or read-wait */
};

    /* state data structure */
struct IIO_MSTATE {
    int channel;			/* the current channel */
};


HIDDEN IIO_STATUS iio_bvmipadc_adc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Read the ADCs as specified, using either polling or read-wait method.
     * It's possible the channel-change and the conversion can be paralleled
     * in some way when doing ranges, but I haven't time to look at it now
     */
    unsigned seqno;

    /* we do only read operations */
    if (op != iio_op_read)
	return iio_error("Operation code not supported by channel");

    /* for each channel in the range */
    for (seqno = first; seqno < first + number; ++seqno) {

	uint16_t val;

	if (seqno != state->channel) 
	    /* change the channel and start conversion */
	    *reg->chselcnv = state->channel = seqno;
	else
	    /* just start a conversion (no interrupt) */
	    *reg->chcnv = 0;

	/* read the ADC */
	if (reg->poll) {
	    /* spin-lock read should have a timeout */
	    while ((val = *reg->adcst) & 0x8000) ;
	} else
	    /* read--lock bus until complete */
	    val = *reg->adcimm;

	/* return result */
	iio_eret( iio_data_set(opnode, seqno, (int)val) );
    }

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_bvmipadc_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialise the module using the configuration already loaded in the
     * state and register structures. There should be no need to probe,
     * as the IP carrier driver has already done it. Just initialise the
     * registers to match the state (or vice-versa).
     */

    /* check our identity */
    iio_eret( iio_ipinfo_ident(reg->slot, 0xe7, 0x21) );

    /* set the current channel to a nonsense value */
    state->channel = -1;

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_bvmipadc_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the VME base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_CHNODE *chnode;
    IIO_MREG *reg;
    int chan;
    double gain = 1.0;
    IIO_BOOL cloop = iio_bool_false;


    /* get a zeroed register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get a state structure */
    iio_eret( iio_module_state(module, sizeof(IIO_MSTATE)) );

    /* get the arguments */
    iio_eret( iio_arg(argv, "slot", iio_arg_channel, &reg->slot) );
    iio_eret( iio_arg(argv, "gain", iio_arg_double, &gain) );
    iio_eret( iio_arg(argv, "cloop", iio_arg_bool, &cloop) );
    iio_eret( iio_arg(argv, "poll", iio_arg_bool, &reg->poll) );

    if (!reg->slot)
	return iio_error("No slot specified");

    /* resolve local register addresses */
    iio_eret(
	iio_resolve_list(
	    reg->slot, iio_space_io,
	    iio_size_16, 0x0, &reg->adcst,
	    iio_size_16, 0x0, &reg->chselcnv,
	    iio_size_16, 0x2, &reg->adcimm,
	    iio_size_16, 0x2, &reg->chcnv,
	    iio_size_16, 0x4, &reg->intstat,
	    iio_size_16, 0x4, &reg->intvec,
	    iio_size_16, 0x6, &reg->promcntl,
	    0
	)
    );

    /* register the channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 12, NCHAN,
	    iio_bvmipadc_adc, &chnode
	)
    );

    /* set the gain conversions for all channels */
    for (chan = 0; chan < NCHAN; ++chan)
	iio_eret(
	    iio_chnode_linear(
		chnode, chan,
		5.0/4096.0 * gain/(cloop ? 250.0 : 1.0),
		0.0,
		(cloop) ? "A" : "V"
	    )
	);

    return iio_status_ok;
}


IIO_STATUS iio_bvmipadc(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    return iio_minfo(
	"bvmipadc",
	"BVM IP-ADC",
	"$Revision: 2222 $",
	iio_multi_yes,
	iio_bvmipadc_install,
	iio_bvmipadc_init
    );
}
