/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id $
 *
 * automio - driver for the Automation group PC104 digital I/O card
 *
 *	-bus <channel>			ISA bus (default isa.0)
 *	-address <addr>			bus address in ISA memory space
 *                                      (default 0xD0000)
 *
 *      This board provides 32 bytes of digital I/O as follows:
 *             Byte              Function
 *             ----              --------
 *             0-9               Digital inputs             (RO)
 *            10-31              Registers                  (R/W)
 *
 */

#include "../internal.h"

    /* default base address */
#define BASE 0xD0000

    /* list of allowable base addresses */
#define NBASES 10
uint32_t iio_automio_bases[] = {
    0xC8000, 0xCC000, 0xD0000, 0xD4000, 0xD8000, 0xDC000, 0xE0000,
    0xE4000, 0xE8000, 0xEC000
};

    /* amount ISA address space occluded (bytes) */
#define SIZE		32


    /* register pointer structure */
struct IIO_MREG {
    IIO bus;			/* channel for the ISA bus */
    /* 32 digital I/O registers */
    volatile uint8_t *diginp[10];
    volatile uint8_t *digio[22];

    /* these are used for the address resolution */
    uint32_t base;		/* ISA bus board base address (shunt E6-E7) */
};


HIDDEN IIO_STATUS iio_automio_dio(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->digio[seqno] = (uint8_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->digio[seqno] & 0xff) );
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->digio[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automio_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->diginp[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automio_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * The AUTOMIO board has 32 memory mapped registers.
     * Probing an ISA bus is a waste of time, as there
     * is no bus error mechanism.
     */

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automio_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    int i, isValidBaseAddress = 0;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    reg->base = BASE;

    /* decode the arguments */
    iio_eret( iio_arg(argv, "address", iio_arg_uint32, &reg->base) );
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &reg->bus) );

    /* open ISA channel is not specified (usual) */
    if (!reg->bus)
	iio_eret( iio_open("isa.0", 0, &reg->bus) );

    /* Check for valid address */
    for (i=0; i<NBASES; i++)
        if (iio_automio_bases[i] == reg->base) {
	    isValidBaseAddress = 1;
	    break;
	}
    if (! isValidBaseAddress)
        return iio_error("Illegal address: should be on a 16K"
			 " boundary between 0xC8000-0xEC000");

    /* do one mapping for the board */
    iio_eret( iio_map(reg->bus, iio_space_mem, reg->base, SIZE));

    /* resolve the board registers */
    iio_eret(
	iio_resolve_list(
	    reg->bus, iio_space_mem,
	    /* Digital input registers */
	    iio_size_8, reg->base + 0, &reg->diginp[0],
	    iio_size_8, reg->base + 1, &reg->diginp[1],
	    iio_size_8, reg->base + 2, &reg->diginp[2],
	    iio_size_8, reg->base + 3, &reg->diginp[3],
	    iio_size_8, reg->base + 4, &reg->diginp[4],
	    iio_size_8, reg->base + 5, &reg->diginp[5],
	    iio_size_8, reg->base + 6, &reg->diginp[6],
	    iio_size_8, reg->base + 7, &reg->diginp[7],
	    iio_size_8, reg->base + 8, &reg->diginp[8],
	    iio_size_8, reg->base + 9, &reg->diginp[9],

	    /* Digital registers */
	    iio_size_8, reg->base + 10, &reg->digio[0],
	    iio_size_8, reg->base + 11, &reg->digio[1],
	    iio_size_8, reg->base + 12, &reg->digio[2],
	    iio_size_8, reg->base + 13, &reg->digio[3],
	    iio_size_8, reg->base + 14, &reg->digio[4],
	    iio_size_8, reg->base + 15, &reg->digio[5],
	    iio_size_8, reg->base + 16, &reg->digio[6],
	    iio_size_8, reg->base + 17, &reg->digio[7],
	    iio_size_8, reg->base + 18, &reg->digio[8],
	    iio_size_8, reg->base + 19, &reg->digio[9],
	    iio_size_8, reg->base + 20, &reg->digio[10],
	    iio_size_8, reg->base + 21, &reg->digio[11],
	    iio_size_8, reg->base + 22, &reg->digio[12],
	    iio_size_8, reg->base + 23, &reg->digio[13],
	    iio_size_8, reg->base + 24, &reg->digio[14],
	    iio_size_8, reg->base + 25, &reg->digio[15],
	    iio_size_8, reg->base + 26, &reg->digio[16],
	    iio_size_8, reg->base + 27, &reg->digio[17],
	    iio_size_8, reg->base + 28, &reg->digio[18],
	    iio_size_8, reg->base + 29, &reg->digio[19],
	    iio_size_8, reg->base + 30, &reg->digio[20],
	    iio_size_8, reg->base + 31, &reg->digio[21],

	    0
	)
    );

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_di, 8, 10,
	    iio_automio_di, NULL
	)
    );
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_dio, 8, 22,
	    iio_automio_dio, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_automio(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "automio",
	    "Automation Group Digital I/O Board (PC104)",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_automio_install,
	    iio_automio_init
	)
    );
    return iio_status_ok;
}
