/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Manufacturing Science and Technology
 * $Id $
 *
 * automgenio - driver for the Automation group PC104 generic I/O card
 *
 *	-bus <channel>			ISA bus (default isa.0)
 *	-address <addr>			bus address in ISA memory space
 *                                      (default 0xD0000)
 *
 *      This board is a very general memory-mapped I/O module. The memory
 *      map starts with a header that describes the hardware available on
 *      the board. The actual hardware registers follow this header.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include "../internal.h"

    /* default base address */
#define BASE 0xD0000

    /* list of allowable base addresses */
#define NBASES 10
static uint32_t iio_automio_bases[] = {
    0xC8000, 0xCC000, 0xD0000, 0xD4000, 0xD8000, 0xDC000, 0xE0000,
    0xE4000, 0xE8000, 0xEC000
};

    /* amount ISA address space occluded (bytes) */
#define SIZE		256


    /* register pointer structure */
struct IIO_MREG {
    IIO bus;			/* channel for the ISA bus */
    /* I/O registers */
    volatile uint8_t ** di8;
    int numDi8;
    volatile uint16_t ** di16;
    int numDi16;
    volatile uint32_t ** di32;
    int numDi32;
    volatile uint64_t ** di64;
    int numDi64;
    volatile uint8_t ** do8;
    int numDo8;
    volatile uint16_t ** do16;
    int numDo16;
    volatile uint32_t ** do32;
    int numDo32;
    volatile uint64_t ** do64;
    int numDo64;
    volatile uint8_t ** adc8;
    int numAdc8;
    volatile uint16_t ** adc16;
    int numAdc16;
    volatile uint32_t ** adc32;
    int numAdc32;
    volatile uint64_t ** adc64;
    int numAdc64;
    volatile uint8_t ** dac8;
    int numDac8;
    volatile uint16_t ** dac16;
    int numDac16;
    volatile uint32_t ** dac32;
    int numDac32;
    volatile uint64_t ** dac64;
    int numDac64;
    volatile uint8_t ** ctr8;
    int numCtr8;
    volatile uint16_t ** ctr16;
    int numCtr16;
    volatile uint32_t ** ctr32;
    int numCtr32;
    volatile uint64_t ** ctr64;
    int numCtr64;
    volatile uint8_t ** reg8;
    int numReg8;
    volatile uint16_t ** reg16;
    int numReg16;
    volatile uint32_t ** reg32;
    int numReg32;
    volatile uint64_t ** reg64;
    int numReg64;
    /* these are used for the address resolution */
    uint32_t base;		/* ISA bus board base address */
    volatile uint8_t * headerBase;
    uint32_t ioBase;	/* ISA bus base address for I/O */
    int boardRev;
};


HIDDEN IIO_STATUS iio_automgenio_di8(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->di8[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_di16(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->di16[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_di32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->di32[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_di64(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->di64[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_do8(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->do8[seqno] = (uint8_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->do8[seqno] & 0xff) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_do16(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->do16[seqno] = (uint16_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->do16[seqno] & 0xffff) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_do32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->do32[seqno] = (uint32_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->do32[seqno]) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_do64(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->do64[seqno] = (uint64_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->do64[seqno]) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_adc8(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->adc8[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_adc16(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->adc16[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_adc32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->adc32[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_adc64(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->adc64[seqno]) );
	    break;

	case iio_op_write:
	case iio_op_readback:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_dac8(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->dac8[seqno] = (uint8_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->dac8[seqno] & 0xff) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_dac16(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->dac16[seqno] = (uint16_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->dac16[seqno] & 0xffff) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_dac32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->dac32[seqno] = (uint32_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->dac32[seqno]) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_dac64(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->dac64[seqno] = (uint64_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_readback:
	    iio_eret( iio_data_set(opnode, seqno, *reg->dac64[seqno]) );
	    break;

	case iio_op_read:
	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_ctr8(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->ctr8[seqno] = (uint8_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->ctr8[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_ctr16(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->ctr16[seqno] = (uint16_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->ctr16[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_ctr32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->ctr32[seqno] = (uint32_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->ctr32[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_ctr64(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->ctr64[seqno] = (uint64_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->ctr64[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_reg8(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->reg8[seqno] = (uint8_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->reg8[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_reg16(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->reg16[seqno] = (uint16_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->reg16[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_reg32(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->reg32[seqno] = (uint32_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->reg32[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_reg64(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    unsigned seqno;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_write:
	    *reg->reg64[seqno] = (uint64_t)iio_data_get(opnode, seqno);
	    break;

	case iio_op_read:
	    iio_eret( iio_data_set(opnode, seqno, *reg->reg64[seqno]) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * The AUTOMIO board has 32 memory mapped registers.
     * Probing an ISA bus is a waste of time, as there
     * is no bus error mechanism.
     */

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_automgenio_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;
    int i, j, k, isValidBaseAddress = 0;
    unsigned char header;
    int headerLen, headerType;
    unsigned int objByte;
    int objWidth, objType, objNum;
    int curOffset;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    reg->base = BASE;

    /* decode the arguments */
    iio_eret( iio_arg(argv, "address", iio_arg_uint32, &reg->base) );
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &reg->bus) );

    /* open ISA channel is not specified (usual) */
    if (!reg->bus)
	iio_eret( iio_open("isa.0", 0, &reg->bus) );

    /* Check for valid address */
    for (i=0; i<NBASES; i++)
        if (iio_automio_bases[i] == reg->base) {
	    isValidBaseAddress = 1;
	    break;
	}
    if (! isValidBaseAddress)
        return iio_error("Illegal address: should be on a 16K"
			 " boundary between 0xC8000-0xEC000");

    /* do one mapping for the board */
    iio_eret( iio_map(reg->bus, iio_space_mem, reg->base, SIZE));

    /* resolve the header base register */
    iio_eret(
	iio_resolve(
	    reg->bus, iio_space_mem, iio_size_8, reg->base + 0,
	    (void **) &reg->headerBase
	    )
	);

    /* read and interpret the header */
    header = * reg->headerBase;
    headerType = header >> 5;
    headerLen = header & 0x1F;
    curOffset = headerLen;
    if (curOffset % 2)
        curOffset++;
    reg->boardRev = * (reg->headerBase + 1);
/*
    printf ("Rev = %d, Type = %d, Header len = %d\n", reg->boardRev,
	    headerType, headerLen);
    printf ("obj offset = %d\n", curOffset);
*/
    /* resolve the board registers according to the header */
    for (i=2; i<headerLen; i++) {
        objByte = * (reg->headerBase + i);
	objNum = objByte & 0x07;
	objType = (objByte >> 3) & 0x07;
	objWidth = objByte >> 6;
/*
	printf ("Object %d: width = %d, type = %d, number = %d\n",
		i-1, objWidth, objType, objNum);
	printf ("curOffset = %d\n", curOffset);
*/
	if (objNum == 0)
	    continue;
	switch (objType) {
	    case 0 : /* Digital input */
	        switch (objWidth) {
		    case 0 : /* 8-bits */
		        if (objNum % 2)
			    return (iio_error ("odd number of "
					       "8-bit objects"
					       " not allowed"));
			if (reg->di8 == NULL) {
			    if ((reg->di8 = (volatile uint8_t **) 
				     malloc (objNum *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->di8 = (volatile uint8_t **) 
				     realloc (reg->di8, (objNum +
							 reg->numDi8) *
					      sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDi8, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_8, reg->base + curOffset,
				    (void **) &reg->di8[j]));
			    curOffset++;
			}
			reg->numDi8 += objNum;
		        break;
		    case 1 : /* 16-bits */
			if (reg->di16 == NULL) {
			    if ((reg->di16 = (volatile uint16_t **) 
				     malloc (objNum *
					     sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->di16 = (volatile uint16_t **) 
				     realloc (reg->di16, (objNum + 
						      reg->numDi16) *
					      sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDi16, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_16, reg->base + curOffset,
				    (void **) &reg->di16[j]));
			    curOffset += 2;
			}
			reg->numDi16 += objNum;
		        break;
		    case 2 : /* 32-bits */
			if (reg->di32 == NULL) {
			    if ((reg->di32 = (volatile uint32_t **) 
				     malloc (objNum *
					     sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->di32 = (volatile uint32_t **) 
				     realloc (reg->di32, (objNum +
						      reg->numDi32) *
					      sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDi32, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_32, reg->base + curOffset,
				    (void **) &reg->di32[j]));
			    curOffset += 4;
			}
			reg->numDi32 += objNum;
		        break;
		    case 3 : /* 64-bits */
			if (reg->di64 == NULL) {
			    if ((reg->di64 = (volatile uint64_t **) 
				     malloc (objNum *
					     sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->di64 = (volatile uint64_t **) 
				     realloc (reg->di64, (objNum + 
						      reg->numDi64) *
					      sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDi64, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_64, reg->base + curOffset,
				    (void **) &reg->di64[j]));
			    curOffset += 8;
			}
			reg->numDi64 += objNum;
		        break;
		}
	        break;
	    case 1 : /* Digital output */
	        switch (objWidth) {
		    case 0 : /* 8-bits */
		        if (objNum % 2)
			    return (iio_error ("odd number of "
					       "8-bit objects"
					       " not allowed"));
			if (reg->do8 == NULL) {
			    if ((reg->do8 = (volatile uint8_t **) 
				     malloc (objNum *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->do8 = (volatile uint8_t **) 
				     realloc (reg->do8, (objNum +
							 reg->numDo8) *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDo8, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_8, reg->base + curOffset,
				    (void **) &reg->do8[j]));
			    curOffset++;
			}
			reg->numDo8 += objNum;
		        break;
		    case 1 : /* 16-bits */
			if (reg->do16 == NULL) {
			    if ((reg->do16 = (volatile uint16_t **) 
				     malloc (objNum *
					     sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->do16 = (volatile uint16_t **) 
				     realloc (reg->do16, (objNum +
						     reg->numDo16) *
					      sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDo16, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_16, reg->base + curOffset,
				    (void **) &reg->do16[j]));
			    curOffset += 2;
			}
			reg->numDo16 += objNum;
		        break;
		    case 2 : /* 32-bits */
			if (reg->do32 == NULL) {
			    if ((reg->do32 = (volatile uint32_t **) 
				     malloc (objNum *
					     sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->do32 = (volatile uint32_t **) 
				     realloc (reg->do32, (objNum +
						      reg->numDo32) *
					      sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDo32, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_32, reg->base + curOffset,
				    (void **) &reg->do32[j]));
			    curOffset += 4;
			}
			reg->numDo32 += objNum;
		        break;
		    case 3 : /* 64-bits */
			if (reg->do64 == NULL) {
			    if ((reg->do64 = (volatile uint64_t **) 
				     malloc (objNum *
					     sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->do64 = (volatile uint64_t **) 
				     realloc (reg->do64, (objNum + 
						       reg->numDo64) *
					      sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDo64, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_64, reg->base + curOffset,
				    (void **) &reg->do64[j]));
			    curOffset += 8;
			}
			reg->numDo64 += objNum;
		        break;
		}
	        break;
	    case 2 : /* ADC */
	        switch (objWidth) {
		    case 0 : /* 8-bits */
		        if (objNum % 2)
			    return (iio_error ("odd number of "
					       "8-bit objects"
					       " not allowed"));
			if (reg->adc8 == NULL) {
			    if ((reg->adc8 = (volatile uint8_t **) 
				     malloc (objNum *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->adc8 = (volatile uint8_t **) 
				     realloc (reg->adc8, (objNum + reg->numAdc8) *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numAdc8, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_8, reg->base + curOffset,
				    (void **) &reg->adc8[j]));
			    curOffset++;
			}
			reg->numAdc8 += objNum;
		        break;
		    case 1 : /* 16-bits */
			if (reg->adc16 == NULL) {
			    if ((reg->adc16 = (volatile uint16_t **) 
				     malloc (objNum *
					     sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->adc16 = (volatile uint16_t **) 
				     realloc (reg->adc16, (objNum + reg->numAdc16) *
					      sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numAdc16, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_16, reg->base + curOffset,
				    (void **) &reg->adc16[j]));
			    curOffset += 2;
			}
			reg->numAdc16 += objNum;
		        break;
		    case 2 : /* 32-bits */
			if (reg->adc32 == NULL) {
			    if ((reg->adc32 = (volatile uint32_t **) 
				     malloc (objNum *
					     sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->adc32 = (volatile uint32_t **) 
				     realloc (reg->adc32, (objNum + reg->numAdc32) *
					      sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numAdc32, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_32, reg->base + curOffset,
				    (void **) &reg->adc32[j]));
			    curOffset += 4;
			}
			reg->numAdc32 += objNum;
		        break;
		    case 3 : /* 64-bits */
			if (reg->adc64 == NULL) {
			    if ((reg->adc64 = (volatile uint64_t **) 
				     malloc (objNum *
					     sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->adc64 = (volatile uint64_t **) 
				     realloc (reg->adc64, (objNum + reg->numAdc64) *
					      sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numAdc64, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_64, reg->base + curOffset,
				    (void **) &reg->adc64[j]));
			    curOffset += 8;
			}
			reg->numAdc64 += objNum;
		        break;
		}
	        break;
	    case 3 : /* DAC */
	        switch (objWidth) {
		    case 0 : /* 8-bits */
		        if (objNum % 2)
			    return (iio_error ("odd number of "
					       "8-bit objects"
					       " not allowed"));
			if (reg->dac8 == NULL) {
			    if ((reg->dac8 = (volatile uint8_t **) 
				     malloc (objNum *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->dac8 = (volatile uint8_t **) 
				     realloc (reg->dac8, (objNum + reg->numDac8) *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDac8, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_8, reg->base + curOffset,
				    (void **) &reg->dac8[j]));
			    curOffset++;
			}
			reg->numDac8 += objNum;
		        break;
		    case 1 : /* 16-bits */
			if (reg->dac16 == NULL) {
			    if ((reg->dac16 = (volatile uint16_t **) 
				     malloc (objNum *
					     sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->dac16 = (volatile uint16_t **) 
				     realloc (reg->dac16, (objNum + reg->numDac16) *
					      sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDac16, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_16, reg->base + curOffset,
				    (void **) &reg->dac16[j]));
			    curOffset += 2;
			}
			reg->numDac16 += objNum;
		        break;
		    case 2 : /* 32-bits */
			if (reg->dac32 == NULL) {
			    if ((reg->dac32 = (volatile uint32_t **) 
				     malloc (objNum *
					     sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->dac32 = (volatile uint32_t **) 
				     realloc (reg->dac32, (objNum + reg->numDac32) *
					      sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDac32, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_32, reg->base + curOffset,
				    (void **) &reg->dac32[j]));
			    curOffset += 4;
			}
			reg->numDac32 += objNum;
		        break;
		    case 3 : /* 64-bits */
			if (reg->dac64 == NULL) {
			    if ((reg->dac64 = (volatile uint64_t **) 
				     malloc (objNum *
					     sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->dac64 = (volatile uint64_t **) 
				     realloc (reg->dac64, (objNum + reg->numDac64) *
					      sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numDac64, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_64, reg->base + curOffset,
				    (void **) &reg->dac64[j]));
			    curOffset += 8;
			}
			reg->numDac64 += objNum;
		        break;
		}
	        break;
	    case 4 : /* Counter */
	        switch (objWidth) {
		    case 0 : /* 8-bits */
		        if (objNum % 2)
			    return (iio_error ("odd number of "
					       "8-bit objects"
					       " not allowed"));
			if (reg->ctr8 == NULL) {
			    if ((reg->ctr8 = (volatile uint8_t **) 
				     malloc (objNum *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->ctr8 = (volatile uint8_t **) 
				     realloc (reg->ctr8, (objNum + reg->numCtr8) *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numCtr8, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_8, reg->base + curOffset,
				    (void **) &reg->ctr8[j]));
			    curOffset++;
			}
			reg->numCtr8 += objNum;
		        break;
		    case 1 : /* 16-bits */
			if (reg->ctr16 == NULL) {
			    if ((reg->ctr16 = (volatile uint16_t **) 
				     malloc (objNum *
					     sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->ctr16 = (volatile uint16_t **) 
				     realloc (reg->ctr16, (objNum + reg->numCtr16) *
					      sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numCtr16, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_16, reg->base + curOffset,
				    (void **) &reg->ctr16[j]));
			    curOffset += 2;
			}
			reg->numCtr16 += objNum;
		        break;
		    case 2 : /* 32-bits */
			if (reg->ctr32 == NULL) {
			    if ((reg->ctr32 = (volatile uint32_t **) 
				     malloc (objNum *
					     sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->ctr32 = (volatile uint32_t **) 
				     realloc (reg->ctr32, (objNum + reg->numCtr32) *
					      sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numCtr32, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_32, reg->base + curOffset,
				    (void **) &reg->ctr32[j]));
			    curOffset += 4;
			}
			reg->numCtr32 += objNum;
		        break;
		    case 3 : /* 64-bits */
			if (reg->ctr64 == NULL) {
			    if ((reg->ctr64 = (volatile uint64_t **) 
				     malloc (objNum *
					     sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->ctr64 = (volatile uint64_t **) 
				     realloc (reg->ctr64, (objNum + reg->numCtr64) *
					      sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numCtr64, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_64, reg->base + curOffset,
				    (void **) &reg->ctr64[j]));
			    curOffset += 8;
			}
			reg->numCtr64 += objNum;
		        break;
		}
	        break;
	    case 5 : /* Register */
	        switch (objWidth) {
		    case 0 : /* 8-bits */
		        if (objNum % 2)
			    return (iio_error ("odd number of "
					       "8-bit objects"
					       " not allowed"));
			if (reg->reg8 == NULL) {
			    if ((reg->reg8 = (volatile uint8_t **) 
				     malloc (objNum *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->reg8 = (volatile uint8_t **) 
				     realloc (reg->reg8, (objNum + reg->numReg8) *
					     sizeof (uint8_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numReg8, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_8, reg->base + curOffset,
				    (void **) &reg->reg8[j]));
			    curOffset++;
			}
			reg->numReg8 += objNum;
		        break;
		    case 1 : /* 16-bits */
			if (reg->reg16 == NULL) {
			    if ((reg->reg16 = (volatile uint16_t **) 
				     malloc (objNum *
					     sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->reg16 = (volatile uint16_t **) 
				     realloc (reg->reg16, (objNum + reg->numReg16) *
					      sizeof (uint16_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numReg16, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_16, reg->base + curOffset,
				    (void **) &reg->reg16[j]));
			    curOffset += 2;
			}
			reg->numReg16 += objNum;
		        break;
		    case 2 : /* 32-bits */
			if (reg->reg32 == NULL) {
			    if ((reg->reg32 = (volatile uint32_t **) 
				     malloc (objNum *
					     sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->reg32 = (volatile uint32_t **) 
				     realloc (reg->reg32, (objNum + reg->numReg32) *
					      sizeof (uint32_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numReg32, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_32, reg->base + curOffset,
				    (void **) &reg->reg32[j]));
			    curOffset += 4;
			}
			reg->numReg32 += objNum;
		        break;
		    case 3 : /* 64-bits */
			if (reg->reg64 == NULL) {
			    if ((reg->reg64 = (volatile uint64_t **) 
				     malloc (objNum *
					     sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			} else {
			    if ((reg->reg64 = (volatile uint64_t **) 
				     realloc (reg->reg64, (objNum + reg->numReg64) *
					      sizeof (uint64_t *)))
				    == NULL)
			        return (iio_error ("calloc failed"));
			}
			for (j=reg->numReg64, k=0; k<objNum; k++, j++) {
			    iio_eret(
				iio_resolve(
				    reg->bus, iio_space_mem,
				    iio_size_64, reg->base + curOffset,
				    (void **) &reg->reg64[j]));
			    curOffset += 8;
			}
			reg->numReg64 += objNum;
		        break;
		}
	        break;
	    default : /* digital input */
	        return (iio_error ("invalid object type"));
	        break;
	}
    }

    /*
    printf ("finalOffset = %d\n", curOffset);
    printf ("base = %08X\n", reg->headerBase);
    printf ("do0 = %08X\n", reg->do8[0]);
    printf ("ctr0 = %08X\n", reg->ctr16[0]);
    printf ("reg0 = %08X\n", reg->reg8[0]);
    */
    
    /* register IP channels with the channel list */

    if (reg->numDi8)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_di, 8, reg->numDi8,
			    iio_automgenio_di8, NULL
			    )
		 );
    if (reg->numDi16)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_di, 16, reg->numDi16,
			    iio_automgenio_di16, NULL
			    )
		 );
    if (reg->numDi32)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_di, 32, reg->numDi32,
			    iio_automgenio_di32, NULL
			    )
		 );
    if (reg->numDi64)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_di, 64, reg->numDi64,
			    iio_automgenio_di64, NULL
			    )
		 );
    if (reg->numDo8)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_do, 8, reg->numDo8,
			    iio_automgenio_do8, NULL
			    )
		 );
    if (reg->numDo16)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_do, 16, reg->numDo16,
			    iio_automgenio_do16, NULL
			    )
		 );
    if (reg->numDo32)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_do, 32, reg->numDo32,
			    iio_automgenio_do32, NULL
			    )
		 );
    if (reg->numDo64)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_do, 64, reg->numDo64,
			    iio_automgenio_do64, NULL
			    )
		 );
    if (reg->numAdc8)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_adc, 8, reg->numAdc8,
			    iio_automgenio_adc8, NULL
			    )
		 );
    if (reg->numAdc16)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_adc, 16, reg->numAdc16,
			    iio_automgenio_adc16, NULL
			    )
		 );
    if (reg->numAdc32)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_adc, 32, reg->numAdc32,
			    iio_automgenio_adc32, NULL
			    )
		 );
    if (reg->numAdc64)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_adc, 64, reg->numAdc64,
			    iio_automgenio_adc64, NULL
			    )
		 );
    if (reg->numDac8)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_dac, 8, reg->numDac8,
			    iio_automgenio_dac8, NULL
			    )
		 );
    if (reg->numDac16)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_dac, 16, reg->numDac16,
			    iio_automgenio_dac16, NULL
			    )
		 );
    if (reg->numDac32)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_dac, 32, reg->numDac32,
			    iio_automgenio_dac32, NULL
			    )
		 );
    if (reg->numDac64)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_dac, 64, reg->numDac64,
			    iio_automgenio_dac64, NULL
			    )
		 );
    if (reg->numCtr8)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_ctr, 8, reg->numCtr8,
			    iio_automgenio_ctr8, NULL
			    )
		 );
    if (reg->numCtr16)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_ctr, 16, reg->numCtr16,
			    iio_automgenio_ctr16, NULL
			    )
		 );
    if (reg->numCtr32)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_ctr, 32, reg->numCtr32,
			    iio_automgenio_ctr32, NULL
			    )
		 );
    if (reg->numCtr64)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_ctr, 64, reg->numCtr64,
			    iio_automgenio_ctr64, NULL
			    )
		 );
    if (reg->numReg8)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_reg, 8, reg->numReg8,
			    iio_automgenio_reg8, NULL
			    )
		 );
    if (reg->numReg16)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_reg, 16, reg->numReg16,
			    iio_automgenio_reg16, NULL
			    )
		 );
    if (reg->numReg32)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_reg, 32, reg->numReg32,
			    iio_automgenio_reg32, NULL
			    )
		 );
    if (reg->numReg64)
        iio_eret(
		 iio_chnode(
			    module,
			    iio_chtype_reg, 64, reg->numReg64,
			    iio_automgenio_reg64, NULL
			    )
		 );

    return iio_status_ok;
}


IIO_STATUS iio_automgenio(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "automgenio",
	    "Automation Group Generic I/O Board (PC104)",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_automgenio_install,
	    iio_automgenio_init
	)
    );
    return iio_status_ok;
}
