/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: atc40.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * atc40.c -- driver for GreenSpring ATC40 and ATC30 ISA IP Carriers,
 *            and the CSIRO/MST ATC10 PC/104 IP Carrier
 *
 * The ATC40 and ATC30 are ISA bus IP carriers providing four and three IP
 * slots respectively addressed from the ISA bus.
 *
 * The ATC10 is the name I have given to the CSIRO/MST single IP carrier
 * for the PC/104 bus (which is logically identical to the ISA bus).
 * This has been designed to be identical to the ATC30/40 but with only
 * one IP slot.
 *
 *	-address <addr>			ISA base address
 *	-bus <channel> 			bus (if not isa.0)
 *	-irq <irq> 			IRQ (ignored)
 *
 * None of the interrupt-related stuff is implemented yet. IP memory can be
 * addressed, but only at the default "fixed" 2k mapping.
 */
#include "../internal.h"


    /* factory default base address (ISA) */
#define BASE		0xfc0000

    /* amount ISA address space occluded (bytes), except for IP memory */
#define SIZE		0x4000		/* 16384 bytes */


    /* register pointer structure */
struct IIO_MREG {
    IIO bus;			/* channel for the ISA bus */
    unsigned nslots;		/* number of IP slots */

    /* pointers to the ATC30/40's own registers */
    uint8_t *mem_map[4];	/* memory base address registers */
    uint8_t *int_vect[4][2];/* interrupt vectors (4 IPs, 2 vectors each) */
    uint8_t *int_read;	/* interuptor vector */
    uint8_t *int_enable;	/* interrupt enable */

    /* these are used for the address resolution */
    uint32_t base;		/* ISA bus board base address (shunt E6-E7) */
    uint8_t irq;		/* ISA bus IRQ we are on (shunt E1-E2) */
};


HIDDEN IIO_STATUS iio_atc40_ip(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * An IP carrier has to support operations to resolve IP local IO, ID, 
     * MEM and INT addresses to virtual. This is done by computing the ISA
     * memory address of the requested local location, then resolving that
     * to a processor physical address by calling the resolve operation on the
     * channel representing the ISA bus.
     */
    unsigned slot;

    for (slot = first; slot < first + number; ++slot) {
	uint32_t offs = (uint32_t)iio_data_get_addr(opnode, slot);
	IIO_SIZE size = (IIO_SIZE)(op & IIO_SIZE_MASK);
	void *result;

	/* 
	 * ISA is little-endian, whereas IPs are bit endian. We assume that
	 * we don't have to do anything for 16-bit accesses, as the hardware
	 * will do the byte-swap, but for byte access we have to flip the 
	 * address up or down one
	 */
	if (size == iio_size_8)
	    offs ^= 0x1;

	switch (op & IIO_SPACE_MASK) {

	case iio_space_io:
	    if (offs > 0x7f)
		return iio_error("IPIO address out of range");
	    result = (void *)(reg->base + 0x100 * slot + 0x00 + offs);
	    iio_eret(
		iio_operate_addr(reg->bus, iio_space_mem | size, &result) 
	    );
	    iio_eret( iio_data_set_addr(opnode, slot, result) );
	    break;

	case iio_space_id:
	    if (offs > 0x7f)
		return iio_error("IPID address out of range");
	    result = (void *)(reg->base + 0x100 * slot + 0x80 + offs);
	    iio_eret(
		iio_operate_addr(reg->bus, iio_space_mem | size, &result) 
	    );
	    iio_eret( iio_data_set_addr(opnode, slot, result) );
	    break;

	case iio_space_mem:
	    if (offs > 0x7ff)
		return iio_error("IPMEM address out of range");
	    result = (void *)(reg->base + 0x800 * slot + offs);
	    iio_eret(
		iio_operate_addr(reg->bus, iio_space_mem | size, &result) 
	    );
	    iio_eret( iio_data_set_addr(opnode, slot, result) );
	    break;

	default:
	    return iio_error("Space code not supported by channel");
	}
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_atc40_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * The ATC40 carrier has a bundle of registers, and we (should) probe
     * for these, although probing an ISA bus is a waste of time, as there
     * is no bus error mechanism.
     *
     * The registers control the memory location of the IP memory, so
     * we set these up to match the options given to the install. 
     */

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_atc40_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Decode the module driver arguments, resolve the ISA base address to
     * logical addresses, build the register structure containing pointers 
     * to the device, and register the channels it provides
     */
    IIO_MREG *reg;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );
    reg->base = BASE;

    /* decide which model/number of slots */
    reg->nslots = argv[2][3] - '0';

    /* decode the arguments */
    iio_eret( iio_arg(argv, "address", iio_arg_uint32, &reg->base) );
    iio_eret( iio_arg(argv, "bus", iio_arg_channel, &reg->bus) );
    iio_eret( iio_arg(argv, "irq", iio_arg_uint8, &reg->irq) );

    /* open ISA channel is not specified (usual) */
    if (!reg->bus)
	iio_eret( iio_open("isa.0", 0, &reg->bus) );

    /* check the IRQ number is valid (zero means disabled) */
    if (
	(reg->irq != 0) &&
	(reg->irq < 3 || reg->irq == 8 || reg->irq == 14 || reg->irq > 15)
    )
	return iio_error("Illegal IRQ number");

    /* do one mapping for the board and IP IO/ID space */
    iio_eret( iio_map(reg->bus, iio_space_mem, reg->base, SIZE));

    /* resolve the board registers */
    iio_eret(
	iio_resolve_list(
	    reg->bus, iio_space_mem,
	     
	    /* interrupt pseudo-vectors */
	    iio_size_8, reg->base + 0x0400, &reg->int_vect[0][0],
	    iio_size_8, reg->base + 0x0402, &reg->int_vect[0][1],
	    iio_size_8, reg->base + 0x0480, &reg->int_vect[1][0],
	    iio_size_8, reg->base + 0x0482, &reg->int_vect[1][1],
	    iio_size_8, reg->base + 0x0500, &reg->int_vect[2][0],
	    iio_size_8, reg->base + 0x0502, &reg->int_vect[2][1],
	    iio_size_8, reg->base + 0x0580, &reg->int_vect[3][0],
	    iio_size_8, reg->base + 0x0582, &reg->int_vect[3][1],

	    /* memory mapping registers */
	    iio_size_8, reg->base + 0x3800, &reg->mem_map[0],
	    iio_size_8, reg->base + 0x3900, &reg->mem_map[1],
	    iio_size_8, reg->base + 0x3a00, &reg->mem_map[2],
	    iio_size_8, reg->base + 0x3b00, &reg->mem_map[3],

	    /* interrupter */
	    iio_size_8, reg->base + 0x3b00, &reg->int_read,
	    iio_size_8, reg->base + 0x3c00, &reg->int_enable,
	    0
	)
    );

    /* register IP channels with the channel list */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_ip, 0,
	    reg->nslots,
	    iio_atc40_ip, NULL
	)
    );

    return iio_status_ok;
}


IIO_STATUS iio_atc40(void) {
    /*
     * Call iio_minfo to register this module with IIO
     */
    iio_eret(
	iio_minfo(
	    "atc10",
	    "CSIRO/MST ATC10 PC104/IP Carrier",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_atc40_install,
	    iio_atc40_init
	)
    );
    iio_eret(
	iio_minfo(
	    "atc30",
	    "GreenSpring ATC30 IP Carrier",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_atc40_install,
	    iio_atc40_init
	)
    );
    iio_eret(
	iio_minfo(
	    "atc40",
	    "GreenSpring ATC40 IP Carrier",
	    "$Revision: 2222 $",
	    iio_multi_yes,
	    iio_atc40_install,
	    iio_atc40_init
	)
    );
    return iio_status_ok;
}
