/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: adam4520.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * adam4520.c -- module driver for an Adam 4000 network
 *
 */
#include "../internal.h"

    /* character constants */
#define EOS	0x00		/* end of string */
#define EOL	0x0d		/* end of line (message) */


    /* the register data structure */
struct IIO_MREG {
    int tty;			/* file descriptor of serial port */
    uint32_t baud;		/* port hardware settings */
    uint8_t baudcode;		/* ADAM baud rate code */
    IIO_BOOL rtscts;		/* flow control setting */
    IIO_BOOL checksum;		/* ADAM checksum enabled */
    IIO_BOOL hz50;		/* 50Hz (60ms) integration */
};


HIDDEN IIO_STATUS iio_adam4520_adam(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * There are only two operations you can to on an adam channel, either
     * send-receive (i.e., command-response), or show
     */
    IIO_STATUS stat;
    unsigned seqno, len;
    uint8_t cksum; 
    char *ubuffer;

    for (seqno = first; seqno < first + number; ++seqno) {

	switch (op) {
	case iio_adam_message:
	    /*
	     * Get the pointer to the message buffer. This contains the
	     * command message and gets the results written into it.
	     */
	    ubuffer = (char *)iio_data_get_addr(opnode, seqno);

	    /* always overwrite the address part of the message */
	    iio_eret( iio_adam_hwrite(seqno, 2, ubuffer + 1) );

	    /* check leading character */
	    switch (ubuffer[0]) {
	    case '%':
		/*
		 * We permit only limited dynamic configuration using %.
		 * The (new) address, the baud rate bytes and the checksum 
		 * and itime bits are overwritten, so really only the type
		 * code and data format is up to the calling module driver
		 */

		/* new address must be same as old */
		iio_eret( iio_adam_hwrite(seqno, 2, ubuffer + 3) );

		/* baud rate must match whole network */
		iio_eret( iio_adam_hwrite(reg->baudcode, 2, ubuffer + 7) );

		/*
		 * checksum and itime bits must match whole network.
		 * ubuffer[10] (data format) is unchanged from driver
		 */
		iio_eret(
		    iio_adam_hwrite(
			(reg->hz50 ? 0x8 : 0x0) |
			(reg->checksum ? 0x4 : 0x0),
			1,
			ubuffer + 9
		    )
		);
		break;

	    case '$':
	    case '#':
	    case '@':
		/* normal commands */
		break;

	    default:
		/* illegal commands */
		return iio_error("Bad ADAM command string");
	    }

	    /* get length, compute checksum */
	    for (cksum = len = 0; ubuffer[len] > ' '; ++len)
	        cksum += ubuffer[len];

	    /* append the checksum if required */
	    if (reg->checksum) {
		iio_eret( iio_adam_hwrite(cksum, 2, ubuffer + len) );
		len += 2;
	    }

	    /* log the sent message */
	    if (opnode->chnode->module->log) {
		ubuffer[len] = EOS;
		iio_log("adam.%d: sending `%s': ", seqno, ubuffer);
	    }

	    /* finally append carriage-return and terminator */
	    ubuffer[len++] = EOL;
	    ubuffer[len] = EOS;

	    /* send it to the module */
	    iio_eret( iio_tty_send(reg->tty, ubuffer, len) );

	    /* wait for a reply */
	    if ((stat = iio_tty_recv(reg->tty, ubuffer, 100, &len, 500))) {
		if (opnode->chnode->module->log)
		    iio_log("receive error\n");
		return stat;
	    }

	    /* log the sent message */
	    if (opnode->chnode->module->log) {
		ubuffer[len - 1] = EOS;
		iio_log("received `%s'\n", ubuffer);
	    }

	    /* check and elide the checksum in the response */
	    if (reg->checksum) {
		int index, rcksum;

		/* compute what the checksum ought to be */
		for (cksum = index = 0; index < len - 3; ++index)
		    cksum += ubuffer[index];
		
		/* test against the digits */
		iio_eret( iio_adam_hread(ubuffer + len - 3, 2, &rcksum) );
		if (cksum != (rcksum & 0xff))
		    return iio_error("Bad checksum");

		/* elide the checksum and carriage return */
		ubuffer[len - 3] = EOS;

	    } else
		/* elide the carriage return */
		ubuffer[len - 1] = EOS;

	    /*
	     * Finally check the return code--there are only a
	     * couple of possibilities
	     */
	    switch (ubuffer[0]) {
	    case '>':
	    case '!':
		break;
	    case '?':
		return iio_error("Invalid ADAM command");
	    default:
		return iio_error("Unknown ADAM response");
	    }

	    /* it's up to the caller to interpret the rest of the result */
	    break;

	case iio_op_show:
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_adam4520_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * Initialise the line characteristics. Since we can't actually
     * communicate with the 4520 itself (it is just a repeater) we
     * can't directly check if the line settings match the unit
     *
     * The module drivers for the units connected to the network will
     * test and configure themselves after this.
     */

    /* set up the port according to settings */
    iio_eret(
	iio_tty_line(
	    reg->tty,
	    reg->baud, 8, 1, iio_ttypar_none,
	    iio_ttyflow_none,
	    iio_ttymap_instrip | iio_ttymap_incrnl,
	    iio_ttyecho_none
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_adam4520_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Install an Adam 4000 network on the given serial port channel.
     * This provides 256 possible addresses for Adam 4000-series modules.
     */
    IIO_MREG *reg;

    /* get a register structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* set default register structure settings */
    reg->tty = 0;
    reg->baud = 9600;
    reg->checksum = iio_bool_false;
    reg->rtscts = iio_bool_false;
    reg->hz50 = iio_bool_false;

    /* get the mandatory tty port */
    iio_eret( iio_arg(argv, "tty", iio_arg_file, &reg->tty) );
    if (!reg->tty)
	return iio_error("No TTY port specified");

    /* get the optional arguments */
    iio_eret(
	iio_arg_list(
	    argv,
	    "baud", iio_arg_uint32, &reg->baud,
	    "rtscts", iio_arg_bool, &reg->rtscts,
	    "checksum", iio_arg_bool, &reg->checksum,
	    "50hz", iio_arg_bool, &reg->hz50,
	    0
	)
    );
    
    /* check the baud rate argument */
    switch (reg->baud) {
    case 1200:  reg->baudcode = 3;  break;
    case 2400:  reg->baudcode = 4;  break;
    case 4800:  reg->baudcode = 5;  break;
    case 9600:  reg->baudcode = 6;  break;
    case 19200: reg->baudcode = 7;  break;
    case 38400:
	/* supported by the 4520, but not the modules */
    default:
	return iio_error("Bad ADAM module baud rate");
    }


    /* register the 256 ADAM channels on this network */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adam, 0, 256,
	    iio_adam4520_adam, NULL
	)
    );
    return iio_status_ok;
}


IIO_STATUS iio_adam4520(void) {
    iio_eret(
	iio_minfo(
	    "adam4520",			/* Our module ID */
	    "Adam 4000 network ",	/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_yes,		/* we can be multiply installed */
	    iio_adam4520_install,	/* the install function */
	    iio_adam4520_init		/* the init function */
	)
    );
    return iio_status_ok;
}
