/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: adam4017.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * adam4017.c -- module driver for an ADAM 4017 and 4018 8-channel ADC
 *
 * Options
 *	-address <chan>		ADAM address channel
 *	-range <code>		Input range code
 *
 * ADAM 4018 support is not tested. It may also work for the 4018M
 * (without the data logging)
 */
#include "../internal.h"

#define NCHAN 8


    /* "register" structure */
struct IIO_MREG {
    IIO aa;			/* ADAM address channel */
    uint8_t irc;		/* voltage range code */
    IIO_BOOL adam4018;		/* if a 4018 module */
};


HIDDEN IIO_STATUS iio_adam4017_adc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Read the ADC by sending the appropriate message to the ADAM 
     * module and decoding the results. Unfortunately, the 4017
     * does NOT have a hexadecimal mode, so we must decode the
     * decimal string it returns
     */
    unsigned seqno; 
    char ubuffer[100];
    void *ubp = (void *)ubuffer;
    int value;

    for (seqno = first; seqno < first + number; ++seqno)
	switch (op) {
	case iio_op_read:

	    /* read the ADC channel */
	    iio_slog(ubuffer, "#xx%x", seqno); 
	    iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );

	    /*
	     * Decode and return the result. The unit sends fixed-point
	     * decimal data, with a format dependent on the input range.
	     * We ignore the decimal point, and turn the digits straight
	     * into an integer
	     */
	    iio_eret( iio_adam_dread(ubuffer + 1, 7, &value) );
	    iio_eret( iio_data_set(opnode, seqno, value) );
	    break;

	default:
	    return iio_error("Operation code not supported by channel");
	}
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_adam4017_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * We check the identity and configuration of the module against that
     * specified in the register structure. 
     */
    char ubuffer[100];
    void *ubp = (void *)ubuffer;
    int irc, prc;

    /* identify module */
    iio_slog(ubuffer, "$xxM");
    iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
    iio_eret( iio_adam_hread(ubuffer + 3, 4, &prc) );
    if (!reg->adam4018 && (prc != 0x4017))
	return iio_error("Module is not an ADAM 4017");
    if (reg->adam4018 && (prc != 0x4018))
	return iio_error("Module is not an ADAM 4018");

    /* the input voltage setting must match the option argument */
    iio_slog(ubuffer, "$xx2");
    iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
    iio_eret( iio_adam_hread(ubuffer + 3, 2, &irc) );
    if (irc != reg->irc)
	return iio_error("ADAM 4017/8 input range does not match module");

    return iio_status_ok;
}


IIO_STATUS iio_adam4017_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Install an ADAM 4017 module
     */
    IIO_MREG *reg;
    IIO_CHNODE *chnode;
    int count;

    /* get a "register" structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get the mandatory ADAM-address */
    iio_eret( iio_arg(argv, "address", iio_arg_channel, &reg->aa) );
    if (!reg->aa)
	return iio_error("No ADAM-address specified");

    /* decide if we are a 4017 or a 4018 */
    reg->adam4018 = (iio_string_cmp(argv[2], "adam4018") == 0)
	? iio_bool_true : iio_bool_false;

    /* get the input range code argument */
    iio_eret( iio_arg(argv, "range", iio_arg_uint8, &reg->irc) );

    /* different ranges are permitted for different modules */
    if (reg->adam4018) {
	if (
	    ((reg->irc > 0x06) && (reg->irc < 0x0e)) ||
	    (reg->irc > 0x14)
	)
	    return iio_error("Bad ADAM 4018 range code");
    } else {
	if ((reg->irc < 0x08) || (reg->irc > 0x0d))
	    return iio_error("Bad ADAM 4017 range code");
    }

    /* register the 8 ADC channels */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 16, NCHAN,
	    iio_adam4017_adc,
	    &chnode
	)
    );

    /* set the input voltage range */
    for (count = 0; count < NCHAN; ++count)
	iio_eret(
	    iio_chnode_linear(
		chnode, count,
		iio_adam_info[reg->irc].escale,
		0.0,
		iio_adam_info[reg->irc].unit
	    )
	);

    return iio_status_ok;
}


IIO_STATUS iio_adam4017(void) {
    iio_eret(
	iio_minfo(
	    "adam4017",			/* Our module ID */
	    "ADAM 4017 ADC module",	/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_yes,		/* we can be multiply installed */
	    iio_adam4017_install,	/* the install function */
	    iio_adam4017_init		/* the init function */
	)
    );
    iio_eret(
	iio_minfo(
	    "adam4018",			/* Our module ID */
	    "ADAM 4018 ADC module",	/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_yes,		/* we can be multiply installed */
	    iio_adam4017_install,	/* the install function */
	    iio_adam4017_init		/* the init function */
	)
    );
    return iio_status_ok;
}
