/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: adam4012.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * adam4012.c -- module driver for an ADAM 4012 and 4011 ADC/switch
 *
 * Options
 *	-address <chan>		ADAM address channel
 *	-range <code>		Input range code
 *
 * ADAM 4011 support is not tested, but is probably OK.
 * The digital event counter and alarm stuff is not implemented; it
 * should be because alarm mode (I think) affects the digital in/out
 * pins
 */
#include "../internal.h"

#define NCHAN 1


    /* "register" structure */
struct IIO_MREG {
    IIO aa;			/* ADAM address channel */
    uint8_t irc;		/* voltage range code */
    enum {			/* subtype of unit */
        iio_ADAM4011,
        iio_ADAM4012,
        iio_ADAM4013
    } type;
};


HIDDEN IIO_STATUS iio_adam4012_adc(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Read the ADC by sending the appropriate message to the ADAM 
     * module and decoding the results. Unfortunately, the 4012
     * does NOT have a hexadecimal mode, so we must decode the
     * decimal string it returns
     */
    char ubuffer[100];
    void *ubp = (void *)ubuffer;
    int value;

    switch (op) {
    case iio_op_read:

	/* read the ADC channel */
	iio_slog(ubuffer, "#xx"); 
	iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
	iio_eret( iio_adam_dread(ubuffer + 1, 7, &value) );
	iio_eret( iio_data_set(opnode, 0, value) );
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_adam4012_do(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Operate on the 2-bit digital output port
     */
    char ubuffer[100];
    void *ubp = (void *)ubuffer;
    int value;

    switch (op) {
    case iio_op_write:
	/* write the DO channel */
	iio_slog(ubuffer, "@xxDO%02x", iio_data_get(opnode, 0));
	iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
	break;

    case iio_op_readback:
	/*
	 * Read back the DO channel: we get !AASOOII where S is the
	 * alarm state (ignored), OO is the digital out state, and
	 * II is the digital in state (ignored)
	 */
	iio_slog(ubuffer, "@xxDI");
	iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
	iio_eret( iio_adam_hread(ubuffer + 4, 2, &value) );
	iio_eret( iio_data_set(opnode, 0, value & 0xff) );
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_adam4012_di(
    IIO_MSTATE *state, IIO_MREG *reg, IIO_OPNODE *opnode,
    IIO_OP op, unsigned first, unsigned number
) {
    /*
     * Operate on the 1-bit digital input port
     */
    char ubuffer[100];
    void *ubp = (void *)ubuffer;
    int value;

    switch (op) {
    case iio_op_read:
	/*
	 * Read the DI channel: we get !AASOOII where S is the
	 * alarm state (ignored), OO is the digital out state (ignored), and
	 * II is the digital in state
	 */
	iio_slog(ubuffer, "@xxDI");
	iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
	iio_eret( iio_adam_hread(ubuffer + 6, 2, &value) );
	iio_eret( iio_data_set(opnode, 0, value & 0xff) );
	break;

    default:
	return iio_error("Operation code not supported by channel");
    }
    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_adam4012_init(IIO_MREG *reg, IIO_MSTATE *state) {
    /*
     * We check the identity and configuration of the module against that
     * specified in the register structure. 
     */
    char ubuffer[100];
    void *ubp = (void *)ubuffer;
    int irc, prc;

    /* identify module */
    iio_slog(ubuffer, "$xxM");
    iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
    iio_eret( iio_adam_hread(ubuffer + 3, 4, &prc) );

    if (reg->type == iio_ADAM4011 && (prc != 0x4011))
	return iio_error("Module is not an ADAM 4011");
    if (reg->type == iio_ADAM4012 && (prc != 0x4012))
	return iio_error("Module is not an ADAM 4012");
    if (reg->type == iio_ADAM4013 && (prc != 0x4013))
	return iio_error("Module is not an ADAM 4013");

    /* the input voltage setting must match the option argument */
    iio_slog(ubuffer, "$xx2");
    iio_eret( iio_operate_addr(reg->aa, iio_adam_message, &ubp) );
    iio_eret( iio_adam_hread(ubuffer + 3, 2, &irc) );
    if (irc != reg->irc)
	return iio_error("ADAM 4011/2 input range does not match module");

    return iio_status_ok;
}


IIO_STATUS iio_adam4012_install(IIO_MODULE *module, char *argv[]) {
    /*
     * Install an ADAM 4012 module
     */
    IIO_MREG *reg;
    IIO_CHNODE *chnode;

    /* get a "register" structure */
    iio_eret( iio_module_reg(module, sizeof(IIO_MREG), &reg) );

    /* get the mandatory ADAM-address */
    iio_eret( iio_arg(argv, "address", iio_arg_channel, &reg->aa) );
    if (!reg->aa)
	return iio_error("No ADAM-address specified");

    /* decide if we are a 4011, 4012 or 4013 */
    if (iio_string_cmp(argv[2], "adam4011") == 0)
	reg->type = iio_ADAM4011;
    else
	if (iio_string_cmp(argv[2], "adam4012") == 0)
	    reg->type = iio_ADAM4012;
	else
	    reg->type = iio_ADAM4013;

    /* get the input range code argument */
    iio_eret( iio_arg(argv, "range", iio_arg_uint8, &reg->irc) );

    /* different ranges are permitted for different modules */
    switch (reg->type) {
    case iio_ADAM4011:
	if (
	    ((reg->irc > 0x06) && (reg->irc < 0x0e)) ||
	    (reg->irc > 0x14)
	)
	    return iio_error("Bad ADAM 4011 range code");
	break;

    case iio_ADAM4012:
	if ((reg->irc < 0x08) || (reg->irc > 0x0d))
	    return iio_error("Bad ADAM 4012 range code");
	break;

    case iio_ADAM4013:
	if ((reg->irc < 0x20) || (reg->irc > 0x29))
	    return iio_error("Bad ADAM 4013 range code");
	break;
    }

    /* register the ADC channel */
    iio_eret(
	iio_chnode(
	    module,
	    iio_chtype_adc, 16, 1,
	    iio_adam4012_adc,
	    &chnode
	)
    );

    /* set the input voltage range */
    iio_eret(
	iio_chnode_linear(
	    chnode, 0,
	    iio_adam_info[reg->irc].escale,
	    0.0,
	    iio_adam_info[reg->irc].unit
	)
    );

    /* register two do channels and one di */
    if (reg->type != iio_ADAM4013) {
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_do, 2, 1,
		iio_adam4012_do,
		&chnode
	    )
	);
	iio_eret(
	    iio_chnode(
		module,
		iio_chtype_di, 1, 1,
		iio_adam4012_di,
		&chnode
	    )
	);
    }
    return iio_status_ok;
}


IIO_STATUS iio_adam4012(void) {
    iio_eret(
	iio_minfo(
	    "adam4011",			/* Our module ID */
	    "ADAM 4011 ADC module",	/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_yes,		/* we can be multiply installed */
	    iio_adam4012_install,	/* the install function */
	    iio_adam4012_init		/* the init function */
	)
    );
    iio_eret(
	iio_minfo(
	    "adam4012",			/* Our module ID */
	    "ADAM 4012 ADC module",	/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_yes,		/* we can be multiply installed */
	    iio_adam4012_install,	/* the install function */
	    iio_adam4012_init		/* the init function */
	)
    );
    iio_eret(
	iio_minfo(
	    "adam4013",			/* Our module ID */
	    "ADAM 4013 ADC module",	/* Our name */
	    "$Revision: 2222 $",		/* File RCS ID */
	    iio_multi_yes,		/* we can be multiply installed */
	    iio_adam4012_install,	/* the install function */
	    iio_adam4012_init		/* the init function */
	)
    );
    return iio_status_ok;
}
