/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: adam.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * adam.c -- conversion functions for ADAM modules
 * Robin Kirkham, January 1997
 */
#include "../internal.h"


IIO_ADAMINFO iio_adam_info[] = {
    /*
     * This range table apparently applies to all ADAM modules,
     * although only certain ranges apply to certain modules. The
     * index is the ADAM range code (usually a hex number)
     *
     * THE HSCALE PARAMETER IS NOT CALCULATED YET
     */

    { /* 00 */  1e-6,   1.0,  "V",  "-15/15mV" },
    { /* 01 */  1e-6,   1.0,  "V",  "-50/50mV" },
    { /* 02 */  10e-6,  1.0,  "V",  "-100/100mV" },
    { /* 03 */  10e-6,  1.0,  "V",  "-500/500mV" },
    { /* 04 */  100e-6, 1.0,  "V",  "-1/1V" },
    { /* 05 */  100e-6, 1.0,  "V",  "-2.5/2.5V" },
    { /* 06 */  1e-6,   1.0,  "A",  "-20/20mA" },
    { /* 07 */  1.0,    1.0,  "",   "unused" },
    { /* 08 */  1e-3,   1.0,  "V",  "-10/10V" },
    { /* 09 */  100e-6, 1.0,  "V",  "-5/5V" },
    { /* 0A */  100e-6, 1.0,  "V",  "-1/1V" },
    { /* 0B */  10e-6,  1.0,  "V",  "-500/500mV" },
    { /* 0C */  10e-6,  1.0,  "V",  "-150/150mV" },
    { /* 0D */  1e-6,   1.0,  "A",  "-20/20mA" },
    { /* 0E */  0.01,   1.0,  "C",  "J-type 0/670C" },
    { /* 0F */  0.1,    1.0,  "C",  "K-type 0/1000C" },
    { /* 10 */  0.01,   1.0,  "C",  "T-type -100/400C" },
    { /* 11 */  0.1,    1.0,  "C",  "E-type 0/1000C" },
    { /* 12 */  0.1,    1.0,  "C",  "R-type 500/1750C" },
    { /* 13 */  0.1,    1.0,  "C",  "S-type 500/1750C" },
    { /* 14 */  0.1,    1.0,  "C",  "B-type 500/1800C" },
    { /* 15 */  1.0,    1.0,  "",   "unused" },
    { /* 16 */  1.0,    1.0,  "",   "unused" },
    { /* 17 */  1.0,    1.0,  "",   "unused" },
    { /* 18 */  1.0,    1.0,  "",   "unused" },
    { /* 19 */  1.0,    1.0,  "",   "unused" },
    { /* 1A */  1.0,    1.0,  "",   "unused" },
    { /* 1B */  1.0,    1.0,  "",   "unused" },
    { /* 1C */  1.0,    1.0,  "",   "unused" },
    { /* 1D */  1.0,    1.0,  "",   "unused" },
    { /* 1E */  1.0,    1.0,  "",   "unused" },
    { /* 1F */  1.0,    1.0,  "",   "unused" },
    { /* 20 */  0.1,    1.0,  "C",  "Platinum (385) -100/100C" },
    { /* 21 */  0.1,    1.0,  "C",  "Platinum (385) 0/100C" },
    { /* 22 */  0.1,    1.0,  "C",  "Platinum (385) 0/200C" },
    { /* 23 */  0.1,    1.0,  "C",  "Platinum (385) 0/600C" },
    { /* 24 */  0.1,    1.0,  "C",  "Platinum (392) -100/100C" },
    { /* 25 */  0.1,    1.0,  "C",  "Platinum (392) 0/100C" },
    { /* 26 */  0.1,    1.0,  "C",  "Platinum (392) 0/200C" },
    { /* 27 */  0.1,    1.0,  "C",  "Platinum (392) 0/600C" },
    { /* 28 */  0.1,    1.0,  "C",  "Nickel -80/100C" },
    { /* 29 */  0.1,    1.0,  "C",  "Nickel (392) 0/100C" },
    { /* 2A */  1.0,    1.0,  "",   "unused" },
    { /* 2B */  1.0,    1.0,  "",   "unused" },
    { /* 2C */  1.0,    1.0,  "",   "unused" },
    { /* 2D */  1.0,    1.0,  "",   "unused" },
    { /* 2E */  1.0,    1.0,  "",   "unused" },
    { /* 2F */  1.0,    1.0,  "",   "unused" },

    { /* 30 */  1e-3,   1.0,  "A",  "0/20mA" },
    { /* 31 */  1e-3,   1.0,  "A",  "4/20mA" },
    { /* 32 */  10e-3,  1.0,  "V",  "0/10V" }
};



IIO_STATUS iio_adam_hread(char *hex, int digits, int *result) {
    /*
     * Decodes the hex string with the given number of digits
     * into result, with sign extension.
     *
     * Not pretty but works. Should use a lookup table
     */
    int value = 0;

    switch (*hex) {
    case '8': case '9': case 'A':
    case 'B': case 'C': case 'D':
    case 'E': case 'F':
	value = -1;
	break;
    }

    for ( ; digits; --digits, ++hex) {
	value <<= 4;

	switch (*hex) {
	case '0': case '1': case '2': 
	case '3': case '4': case '5': 
	case '6': case '7': case '8': 
	case '9':
	    value += (*hex - '0');
	    break;

	case 'A': case 'B': case 'C':
	case 'D': case 'E': case 'F':
	    value += (*hex - 'A' + 0xa);
	    break;

	case 'a': case 'b': case 'c':
	case 'd': case 'e': case 'f':
	    value += (*hex - 'a' + 0xa);
	    break;

	default:
	    return iio_error("Bad hex digit");
	}
    }
    if (result)
	*result = value;
    return iio_status_ok;
}

IIO_STATUS iio_adam_hwrite(int value, int digits, char *string) {
    /*
     * Write out the hex equivalent of value into the given string,
     * always producing the given number of digits. The string is
     * NOT (re)terminated
     */

    for (--digits; digits >= 0; --digits) {
	string[digits] = "0123456789ABCDEF"[value & 0xf];
	value >>= 4;
    }
    return iio_status_ok;
}


IIO_STATUS iio_adam_dread(char *dec, int digits, int *result) {
    /*
     * Decodes the hex string with the given number of digits
     * into result, with sign extension
     *
     * Not pretty but works
     */
    int value = 0;
    int sign = 1;

    for ( ; digits; --digits, ++dec)
	switch (*dec) {
	case '.':
	case '+':
	    break;

	case '-':
	    sign = -1;
	    break;

	case '0': case '1': case '2':
	case '3': case '4': case '5':
	case '6': case '7': case '8':
	case '9':
	    value *= 10;
	    value += *dec - '0';
	    break;
	default:
	    return iio_error("Bad decimal digit");
	}

    if (result)
	*result = value * sign;
    return iio_status_ok;
}
