/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: module.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * module.c -- the installed module list
 * Robin Kirkham, June 1996
 */

#include <stdio.h>

#include "internal.h"


HIDDEN IIO_STATUS iio_module_create(IIO_MINFO *minfo, IIO_MODULE **module) {
    /*
     * This is called from iio_module() to create a module structure for a
     * new module. It creates the shared state and register pointer structures,
     * (which the install function subsequently fills out), the module mutex
     * and so on
     */
    IIO_MODULE *new = NULL, *mod = NULL;

    /* check the incoming arguments */
    if (!minfo || minfo->magic != iio_magic_minfo)
	return iio_fatal("NULL or bad minfo pointer");
    if (!module)
	return iio_fatal("NULL module return pointer");

    /* obtain a new module structure and fill out */
    iio_eret( iio_mem_alloc(sizeof(IIO_MODULE), (void **)&new) );
    new->magic = iio_magic_module;
    new->minfo = minfo;
    new->next = NULL;
    new->seq = 0;

    /* rip through the current module list and compute sequence number */
    for (mod = iio_state->module; mod; mod = mod->next)
	if (iio_string_cmp(new->minfo->ident, mod->minfo->ident) == 0)
	    ++new->seq;

    /* check if the module cannot by multiply installed */
    if (new->minfo->multi == iio_multi_no && new->seq > 0) {
	iio_mem_free(new);
	return iio_error("Module cannot be multiply installed");
    }

    /* create a shared monitor semaphore for this module */
    iio_eret( iio_shmutex_create(&new->mutex) );

    /* return the module struct pointer */
    *module = new;

    return iio_status_ok;
}


HIDDEN IIO_STATUS iio_module_destroy(IIO_MODULE *module) {
    /*
     * Deallocate any memory associated with the given module. The module
     * must not be not in the module list, or in use: this is really only
     * to clean up after a module install failed
     */

    if (module->reg)
	iio_mem_free(module->reg);
#if 0
    /* don't know about deleting shared things */
    if (module->state)
	iio_mem_free(module->reg);
    if (module->mutex)
	iio_shmutex_destroy(module->mutex);
#endif
    iio_mem_free(module);

    return iio_status_ok;
}


HIDDEN int iio_module_cmp(IIO_SLL *s1, IIO_SLL *s2) {
    /*
     * Used by the ordered singly-linked list insert. Returns a positive,
     * zero or negative result depending on whether s1 is greater, equal
     * or less that s2. For the module list the basis is the ident code,
     * then the sequence number
     */
    int cmp;

    /* compare the idents, return lexographic result if different */
    if ((cmp = iio_string_cmp(
	((IIO_MODULE *)s1)->minfo->ident,
	((IIO_MODULE *)s2)->minfo->ident
    )))
	return cmp;
    else
	/* sequence number difference should never be zero */
	return ((IIO_MODULE *)s1)->seq - ((IIO_MODULE *)s2)->seq;
}


IIO_STATUS iio_module(char *argv[]) {
    /*
     * Install a module into the IIO system. The module ident should be
     * in argv[2], and configuration options in the other arguments.
     * argv[2] is looked up in the list of minfo structures, and the install
     * function, also in the minfo structure, is called with the argc/argv
     * arguments. This function initialises the module hardware, and calls
     * iio_chnode() to insert its channel information into the chnode list.
     * Then, iio_module() inserts the module structure into the main module
     * list. ref is the config file line reference
     */
    IIO_STATUS stat;
    IIO_MINFO *minfo;
    IIO_MODULE *module;
    char *alias = NULL;

    /* we must at least have an ident code */
    if (!argv[2] || !*argv[2])
	return iio_fatal("NULL or empty module ident code");

    /* look up the first argument in the list of minfo's */
    if (iio_minfo_find(argv[2], &minfo)) {

	/* not there: this is not a catastrophic error */
	iio_log(
	    "%s: %s: No such module\n",
	    argv[0], argv[2]
	);
	return iio_status_ok;
    }

    /* create a new module entry */
    iio_eret( iio_module_create(minfo, &module) );

    /* check for module-generic arguments */
    iio_arg(argv, "log", iio_arg_bool, &module->log);
    iio_arg(argv, "alias", iio_arg_string, &alias);


    /* call the module install function to initialise the data structures */
    iio_fret( stat = minfo->install(module, argv) );

    if (stat) {
        /* installation failed: log the reason */
	iio_log(
	    "%s: %s: install failed: %s\n",
	    argv[0],
	    argv[2],
	    iio_emessage_get()
	);

	/* purge any channels the module may have registered */

	/* delete the module entry */
	iio_module_destroy(module);

	return stat;
    }

    /*
     * Call the module init function to init the hardware, if there is
     * such a function and the init flag is true. This way, hardware and
     * the state structure will only be initialised by the first IIO
     * process (in a Unix system)
     */
    if (minfo->init && iio_state->init) {

	iio_fret( stat = minfo->init(module->reg, module->state) );

	if (stat) {
	    /* initialisation failed: log the reason */
	    iio_log(
		"%s: %s: init failed: %s\n",
		argv[0],
		argv[2],
		iio_emessage_get()
	    );

	    /* purge any channels the module may have registered */

	    /* delete the module entry */
	    iio_module_destroy(module);

	    return stat;
	}
    } 

    /* blank out the interpreted ident argument */
    argv[2] = iio_arg_blank;
    iio_arg_remnants(argv);


    /* insert into ordered module list */
    iio_sll_insert(
	(IIO_SLL **)&iio_state->module,
	(IIO_SLL *)module,
	iio_module_cmp
    );

    /* add the alias for this module */
    if (alias) {
	char buffer[100];
	iio_slog(buffer, "%s.%d", module->minfo->ident, module->seq);
	iio_eret( iio_alias_insert(iio_atype_module, alias, buffer) );
	iio_mem_free(alias);
    }

    /* log a message */
    if (module->log)
	iio_log(
	    "iio: %s.%d: installed%s\n",
	    module->minfo->ident,
	    module->seq,
	    " and initialised"
	);

    return iio_status_ok;
}


IIO_STATUS iio_module_state(IIO_MODULE *module, unsigned int size) {
    /*
     * Allocate a shared memory state block of the given size, and link it 
     * into the module structure. The pointer to the state block is not
     * returned, as it should not be modified by the install function,
     * but by the init function (to which it is subsequently passed).
     */
    IIO_MSTATE *new;

    /* check parameters */ 
    if (!module || module->magic != iio_magic_module)
	return iio_fatal("NULL or bad module pointer");
    if (size > 10000)
	return iio_fatal("Unreasonable module state structure size");

    /* allocate or obtain shared memory */
    iio_eret( iio_shmem_alloc(size, (void **)&(new)) );

    /* link into module */
    module->state = new;

    return iio_status_ok;
}


IIO_STATUS iio_module_reg(
    IIO_MODULE *module, unsigned int size, IIO_MREG **reg
) {
    /*
     * Allocate a non-shared register pointer block of the given size,
     * link it into the module structure, and return the pointer into reg.
     * This function is called from module driver install functions
     */
    IIO_MREG *new;

    /* check parameters */ 
    if (!module || module->magic != iio_magic_module)
	return iio_fatal("NULL or bad module pointer");
    if (size > 10000)
	return iio_fatal("Unreasonable module register structure size");
    if (!reg)
	return iio_fatal("NULL register structure return pointer");

    /* allocate memory */
    iio_eret( iio_mem_alloc(size, (void **)&(new)) );

    /* link and return */
    module->reg = new;
    *reg = new;

    return iio_status_ok;
}


IIO_STATUS iio_module_find(char *string, IIO_MODULE **module) {
    /*
     * Find the module specified in the string, which must be of the 
     * form <ident>.<seq>. A pointer to the module structure is returned
     * into module. Error status is returned if the requested module
     * does not exist
     */
    IIO_MODULE *mod;
    char *pseq;
    int ilength;
    long lseq;

    /* find the sequence number first */
    if (! (pseq =  iio_string_pbrk(string, ".")))
	return iio_error("Missing '.' in module ident");

    /* compute the length of the ident string */
    ilength = (int)(pseq - string);

    /* read the sequence number, after the . */
    iio_eret( iio_decode_long(++pseq, &lseq) );
    if (lseq < 0)
	return iio_error("Can't have negative module sequence numbers");
    
    /*
     * Search the module list for a module with matching ident, ident string
     * length (as we can't terminate the string for strcmp()) and sequence
     * number. Return it into *module if we find it
     */
    for (mod = iio_state->module; mod; mod = mod->next) 
	if (
	    mod->seq == (unsigned int)lseq &&
	    iio_string_ncmp(mod->minfo->ident, string, ilength) == 0 &&
	    iio_string_len(mod->minfo->ident) == ilength
	) {
	    *module = mod;
	    return iio_status_ok;
	}

    return iio_error("No such module");
}


IIO_STATUS iio_module_show(void) {
    /*
     * TEMPORARY FUNCTION
     * Print the module list
     */
    IIO_MODULE *mod;

    printf(
	"%12s %-3s %10s  %10s  %-3s  %s\n",
	"ident",
	"seq",
	"state",
	"reg",
	"log",
	"model"
    );

    /* for each module entry in the list */
    for (mod = iio_state->module; mod; mod = mod->next) {
	printf(
	    "%12s.%-3d 0x%08x  0x%08x  %-3s  %s\n",
	    mod->minfo->ident,
	    mod->seq,
	    (unsigned)mod->state,
	    (unsigned)mod->reg,
	    mod->log ? "yes" : "no",
	    mod->minfo->model
	);
    }
    return iio_status_ok;
}
