/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: minfo.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * minfo.c -- available module driver minfo list
 * Robin Kirkham, July 1996
 */

#include <stdio.h>

#include "internal.h"


HIDDEN int iio_minfo_cmp(IIO_SLL *s1, IIO_SLL *s2) {
    /*
     * Used by the ordered singly-linked list function iio_sll_insert().
     * The open channel are ordered alphabetically by ident
     */
    return iio_string_cmp(((IIO_MINFO *)s1)->ident, ((IIO_MINFO *)s2)->ident);
}


IIO_STATUS iio_minfo(
    const char *ident, const char *model, const char *version,
    IIO_MULTI multi, IIO_INSTALLFN install, IIO_INITFN init
) {
    /*
     * This is called by the module information functions to register their
     * existance with the IIO library. The basic information is recorded into
     * the minfo list. Later, when modules are to be installed, the minfo list
     * is searched to get the module install function pointers, and other
     * information
     */
    IIO_MINFO *new;

    /* check the parameters */ 
    if (!ident || !*ident)
	return iio_fatal("NULL or empty module ident string");
    if (!model || !*model)
	return iio_fatal("NULL or empty module model string");
    if (!version || !*version)
	return iio_fatal("NULL or empty module driver version string");
    if (!install)
	return iio_fatal("NULL module install function pointer");
    if (!init)
	return iio_fatal("NULL module init function pointer");

    /* allocate a new minfo block and fill out */
    iio_eret( iio_mem_alloc(sizeof(IIO_MINFO), (void **)&new) );
    new->magic = iio_magic_minfo;
    new->ident = (char *)ident;
    new->model = (char *)model;
    new->version = (char *)version;
    new->multi = multi;
    new->install = install;
    new->init = init;

    /* link into alphabetical minfo list */
    iio_eret(
        iio_sll_insert(
            (IIO_SLL **)&iio_state->minfo,
            (IIO_SLL *)new,
            iio_minfo_cmp
        )
    );
    return iio_status_ok;
}


IIO_STATUS iio_minfo_call(IIO_INFOFN infofn[]) {
    /*
     * infofn[] is a NULL-terminated array of pointers to module information
     * functions. This function simple calls these function in order. The
     * functions call back iio_minfo() to actually register themselves.
     * This function is usually passed the builtin module list
     */
    int count;

    if (! infofn)
	return iio_fatal("NULL information function pointer array");

    for (count = 0; infofn[count]; ++count)
	iio_eret( infofn[count]() );
    return iio_status_ok;
}


IIO_STATUS iio_minfo_find(char *ident, IIO_MINFO **minfo) {
    /*
     * Look up the ident code in the minfo list. Put the pointer to the
     * minfo structure in *minfo, or return iio_error
     */
    IIO_MINFO *pminfo;

    if (!ident || !*ident)
	return iio_fatal("NULL or empty ident string");

    for (pminfo = iio_state->minfo; pminfo; pminfo = pminfo->next)
	if (iio_string_cmp(ident, pminfo->ident) == 0) {
	    *minfo = pminfo;
	    return iio_status_ok;
	}

    /* return only error status if not found */
    return iio_error("No such module");
}


IIO_STATUS iio_minfo_show(void) {
    /*
     * Print out a list of all the built-in module drivers,
     * and their minfo information 
     */
    IIO_MINFO *minfo;

    printf(
	"%12s  %10s  %10s  %-3s  %-30.30s %s\n",
	"ident",
	"install",
	"init",
	"mul",
	"model",
	"rev"
    );

    for (minfo = iio_state->minfo; minfo; minfo = minfo->next) {
	char buffer[100];
	char *start, *end;

	/* beautify the RCS $Revision: 2222 $ string */
	iio_string_cpy(buffer, minfo->version);
	if ((start = iio_string_chr(buffer, ':')))
	    start += 2;
	else
	    start = buffer;
	if ((end = iio_string_chr(start, '$')))
	    *end = '\0';

	printf(
	    "%12s  0x%08x  0x%08x  %-3s  %-30.30s %s\n",
	    minfo->ident,
	    (unsigned)minfo->install,
	    (unsigned)minfo->init,
	    (minfo->multi == iio_multi_yes) ? "yes" : "no",
	    minfo->model,
	    start
	);
    }
    return iio_status_ok;
}
