/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: map.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * map.c -- virtual to physical mappings
 * Robin Kirkham, July 1996
 */

#include <stdio.h>

#include "internal.h"


IIO_STATUS iio_map(
    IIO_OPEN *chan, IIO_OP space, unsigned laddr, unsigned lsize
) {
    /*
     * Create a virtual-physical mapping that covers the local address range
     * expressed by laddr and lsize, resolved through the given channel and
     * space operation code.
     *
     * This function is called by module drivers to create a mapping so their
     * registers are visible to the program. They subsequently call
     * iio_resolve() to resolve individual addresses through the map thus
     * created
     */
    void *paddr1 = (void *)laddr;
    void *paddr2 = (void *)(laddr + lsize);

    /* check parameters */ 
    if (!chan || chan->magic != iio_magic_open)
	return iio_fatal("NULL or bad channel pointer");

    /*
     * Use the channel to resolve address to physical. Note that the operation
     * code is the arithmatic OR of the space enum and the space operation code
     */
    iio_eret( iio_operate_addr(chan, space | iio_size_8, &paddr1) );
    iio_eret( iio_operate_addr(chan, space | iio_size_8, &paddr2) );

    /* create a new virtual-physical mapping */ 
    iio_eret( iio_map_new(space, paddr1, (unsigned)(paddr2 - paddr1)) );

    return iio_status_ok;
}


HIDDEN int iio_map_cmp(IIO_SLL *s1, IIO_SLL *s2) {
    /*
     * Used by the ordered singly-linked list function iio_sll_insert().
     * The memory mappings are ordered numerically by physical address,
     * then by size. There should be no duplicates (although this is not
     * a disaster, I suppose)
     */
    int cmp;

    if ((cmp = (((IIO_MAP *)s1)->paddr - ((IIO_MAP *)s2)->paddr)))
	return cmp;
    else
	return (((IIO_MAP *)s1)->size - ((IIO_MAP *)s2)->size);
}


IIO_STATUS iio_map_new(IIO_OP space, void *paddr, unsigned psize) {
    /*
     * Ensure that a contiguous mapping, at least enclosing the given
     * physical address range, exists into this process's virtual memory space.
     *
     * First, the range is compared with extant mappings, which are kept in 
     * a list of IIO_MAP structures. If a mapping exists, nothing further
     * need be done. If not, the OS virtual mapping function is called through
     * iio_shmap_alloc(), and the results inserted into the list.
     *
     * On operating systems that don't do virtual memory, or always have
     * 1:1 mappings (like vxWorks and RTEMS) this whole system is a waste of
     * space. But is is required for LynxOS (which does not offer P to V
     * resolution).
     *
     * This function only return error status if the OS map function fails.
     * This is probably a fatal error
     */
    IIO_MAP *new;
    IIO_MAPTYPE type;
    void *vaddr;

    /* get mapping type from space */
    iio_eret( iio_map_type(space, &type) );
	
    /* look it up in the current list */
    if (iio_map_ptov(type, paddr, psize, &vaddr) == iio_status_ok)
	return iio_status_ok;

    /* not in current list: make a new mapping */

    /* get a new IIO_MAP structure */
    iio_eret( iio_mem_alloc(sizeof(IIO_MAP), (void **)&new) );
    new->magic = iio_magic_map;

    /*
     * Get a mapping of the desired physical range. The actual mapping will
     * probably be larger than that actually required, because MMUs map
     * memory in blocks of typically 4096 bytes. For this reason, the physical
     * address of the base of the mapped block will probably be lower 
     * from the one requested. For the same reason, the actual size will
     * probably be larger. The list only has to record these actual values;
     * the original request values are discarded.
     *
     * For memory maps, we call the OS-dependent function iio_shmap_alloc(),
     * while for port maps we call iio_port_alloc()
     */
    if (type == iio_maptype_memory) {
	iio_eret(
	    iio_shmap_alloc(
		paddr, psize,
		&new->paddr, &new->vaddr, &new->size
	    )
	);
	new->type = iio_maptype_memory;
    } else {
	iio_eret(
	    iio_port_alloc(
		paddr, psize,
		&new->paddr, &new->vaddr, &new->size
	    )
	);
	new->type = iio_maptype_port;
    }

    /* insert into the list of maps, ordered by physical address, then size */
    iio_eret(
        iio_sll_insert(
            (IIO_SLL **)&iio_state->map,
            (IIO_SLL *)new,
            iio_map_cmp
        )
    );

    return iio_status_ok;
}


IIO_STATUS iio_map_ptov(
    IIO_MAPTYPE type, void *paddr, unsigned size, void **vaddr
) {
    /*
     * Look up the given physical address range in the address map, and 
     * return the corresponding virtual address. Return error status if there
     * is no mapping, or the mapping does not cover the entire range. Note
     * that there can be more than one mapping from P to V space; this
     * function always uses the first one in the list, which will be the 
     * smallest one installed.
     *
     * The mapping type (port or memory) must simply match
     */
    IIO_MAP *map;

    /* check return argument */
    if (! vaddr)
	return iio_fatal("NULL virtual address return pointer");

    /* zero-sized maps are silly */
    if (! size)
	return iio_error("Zero size mapping requested");

    /* search the list */
    for (map = iio_state->map; map; map = map->next)
	if (
	    (type == map->type) &&
	    (paddr >= map->paddr) &&
	    (((paddr - 1) + size) <= ((map->paddr - 1) + map->size))
	) {
	    /* found an enclosing mapping */
	    *vaddr = map->vaddr + (paddr - map->paddr);
	    return iio_status_ok;
	}

    return iio_error("No mapping of physical address");
}


IIO_STATUS iio_map_vtop(
    IIO_MAPTYPE type, void *vaddr, unsigned size, void **paddr
) {
    /*
     * Look up the given virtual address range in the address map, and 
     * return the corresponding physical address. Return error status if there
     * is no mapping, or the mapping does not cover the entire range.
     *
     * The mapping type (port or memory) must simply match
     */
    IIO_MAP *map;

    /* check return argument */
    if (! paddr)
	return iio_fatal("NULL physical address return pointer");

    /* search the list */
    for (map = iio_state->map; map; map = map->next)
	if (
	    (type == map->type) &&
	    (vaddr >= map->vaddr) &&
	    (((vaddr - 1) + size) <= ((map->vaddr - 1) + map->size))
	) {
	    /* found an enclosing mapping */
	    *paddr = map->paddr + (vaddr - map->vaddr);
	    return iio_status_ok;
	}

    return iio_error("No mapping to virtual address");
}


IIO_STATUS iio_map_type(IIO_OP space, IIO_MAPTYPE *type) {
    /*
     * Return mapping type (memory or port) for a given address space.
     * This property should perhaps be part of a more general set of
     * address space property structure, but it seems to be the only
     * one at present
     */
    switch (space) {
    case iio_space_port:
	*type = iio_maptype_port;
	return iio_status_ok;

    case iio_space_io:
    case iio_space_id:
    case iio_space_int:
    case iio_space_mem:
    case iio_space_mem16:
    case iio_space_mem24:
    case iio_space_mem32:
	*type = iio_maptype_memory;
	return iio_status_ok;

    default:
	return iio_error("Operation code is not an address space code");
    }
}


IIO_STATUS iio_map_done(void) {
    /*
     * This dismantles the mapping list, and calls iio_shmap_free() for each
     * mapping, releasing it back to the system. It is called from iio_done(),
     * which is called at program exit
     */
    IIO_MAP *map, *next;

    for (map = iio_state->map; map; map = next) {
	next = map->next;
	iio_shmap_free(map->vaddr, map->paddr, map->size);
	iio_mem_free(map);
    }
    return iio_status_ok;
}


IIO_STATUS iio_map_show(void) {
    /*
     * TEMPORARY FUNCTION
     * Print a list of current mappings
     */
    IIO_MAP *map;

    printf(
	"    %4s  %10s  %10s  %10s  %6s\n",
	"type",
	"paddr",
	"vaddr",
	"size",
	"size"
    );
    for (map = iio_state->map; map; map = map->next) 
	printf(
	    "    %-4s  0x%08x  0x%08x  0x%08x  %6d\n",
	    map->type ? "port" : "mem",
	    (uint32_t)map->paddr,
	    (uint32_t)map->vaddr,
	    map->size,
	    map->size
	);

    return iio_status_ok;
}
