/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: init.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * init.c -- initialisation functions
 * Robin Kirkham, June 1996
 */
#include "internal.h"


IIO_STATUS iio_init(IIO_INFOFN list[], IIO_IFLAG flags) {
    /*
     * This must be called before any other IIO function. It only calls the
     * OS-specific init function. This must (a) establish whatever OS-specific
     * services are required, (b) establish a system-wide mutex lock,
     * (c) decide if this call is the initial call to iio_init for this system
     * (i.e., the hardware should be initialised), (d) call the function
     * iio_init_iio() to actually initialise IIO and maybe the hardware,
     * and (d) release the lock.
     *
     * Normally list is the standard module driver list iio_standard[],
     * but it could be a cut-down system specific list. Flags indicates some
     * system-wide flags. Both are passed directly to the OS-specific function
     */

    iio_eret( iio_osinit(list, flags) );
    return iio_status_ok;
}


IIO_STATUS iio_init_iio(
    IIO_INFOFN list[], IIO_IFLAG flags, IIO_BOOL init,
    IIO_TFILE method, ...
) {
    /*
     * This is called ONLY by the OS-specific initialisation function
     * iio_osinit(), which passed the list and flags argument it was passed,
     * the configuration file data and access method, and a flag indicating if
     * this is the first call of IIO in this system.
     *
     * This function must be called inside a system-wide exclusion lock, so
     * that the IIO library and module driver data structures can be
     * constructed consistently
     */
    va_list ap;
    va_start(ap, method);

    /* create and initialiase the global state structure */
    iio_eret( iio_state_init() );

    /* take the process-wide mutex */
    iio_eret( iio_mutex_grab(iio_state->mutex) );

    /* decode the flags argument */
    if (flags & iio_iflag_log)
	iio_state->log = iio_bool_true;

    /*
     * store the init flag in the state structure. This tells the module
     * drivers if they should initialise the hardware or not
     */
    iio_state->init = init;

    /* load driver information into minfo list */
    iio_eret( iio_minfo_call(iio_builtin) );

    /* a hopefully temporary hack */
#if defined(VXWORKS)
    iio_eret( iio_minfo_call(iio_standard) );
#else
    if (list)
	iio_eret( iio_minfo_call(list) );
#endif

    /*
     * Install the null module, so it does not need to be put in the config
     * file all the time. This requires a "fake" config file string
     */
    iio_eret(
	iio_tfile(
	    iio_config_exec,
	    iio_tfile_string,
	    "module null\n",
	    "<internal>"
	)
    );

    /*
     * Parse the configuration file (or string) as a token-file using the
     * execute function iio_config_exec. This calls all the IIO module driver
     * install and init functions as defined in the file 
     */
    iio_eret( iio_tfile_stdarg(iio_config_exec, method, ap) );

    /* release the process-wide mutex */
    iio_eret( iio_mutex_drop(iio_state->mutex) );

    return iio_status_ok;
}
