#ifndef _iio_h_
#define _iio_h_
/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: iio.h 391 2007-02-14 01:49:38Z pra077 $
 *
 * iio.h -- user level function header 
 * Robin Kirkham, June 1996
 */

#ifdef __cplusplus
extern "C" {
#endif

    /* global channel or channel range identifier */
typedef struct IIO_OPEN * IIO;

    /* the config code, used by iio_init() */
typedef enum {
    iio_config_none,		/* no configuration */
    iio_config_stdfile,		/* read configuration from standard file */
    iio_config_file,		/* read configuration from a file */
    iio_config_string		/* read configuration from a string */
} IIO_CONFIG;

    /* the operation code, used by iio_operate() */
typedef enum {
    /* generic operation codes */
    iio_op_nop,			/* no operation */
    iio_op_show,		/* log status of driver/module */
    iio_op_read,		/* read input from the channel */
    iio_op_readback,		/* read previous output from the channel */
    iio_op_write,		/* write output to channel */
    iio_op_clear,		/* write zero to channel */

    /* space resolution codes, always ORed with an IIO_SIZE */
    iio_space_io,		/* resolve input/output space */ 
    iio_space_id,		/* resolve identity space */
    iio_space_int,		/* resolve interrupt space */
    iio_space_mem,		/* resolve memory space */
    iio_space_mem16,		/* resolve 16-bit memory space */
    iio_space_mem24,		/* resolve 24-bit memory space */
    iio_space_mem32,		/* resolve 32-bit memory space */

    /* servo controller codes. These are a bit LM628-specific */
    iio_sc_start,		/* start servo with new target/settings */
    iio_sc_stop,		/* stop motion */
    iio_sc_free,		/* disable servo */
    iio_sc_read_current,	/* read current servo value */
    iio_sc_read_target,		/* read target servo value */
    iio_sc_write_current,	/* write (calibrate) current position */
    iio_sc_write_target,	/* write target position */
    iio_sc_write_target_dt,	/* write trapezoidal dt (e.g. velocity) */
    iio_sc_write_target_ddt,	/* write trapezoidal ddt (e.g. acceleration) */
    iio_sc_read_index,		/* read last index value */
    iio_sc_read_gain_p,		/* read loop proportional gain */
    iio_sc_read_gain_d,		/* read loop derivative gain */
    iio_sc_read_gain_i,		/* read loop integral gain */
    iio_sc_write_gain_p,	/* write loop proportional gain */
    iio_sc_write_gain_d,	/* write loop derivative gain */
    iio_sc_write_gain_i,	/* write loop integral gain */

    /* Adam module codes */
    iio_adam_message,		/* exchange command/reply packet */

    /* more space resolution codes, always ORed with an IIO_SIZE */
    iio_space_port,		/* resolve non-mappable port space */

    IIO_NOPS
} IIO_OP;

    /* init flag type */
typedef enum {
    iio_iflag_none = 0x0,	/* no flags */
    iio_iflag_log = 0x1		/* log some major events */
} IIO_IFLAG;

    /* opening flag type (EXAMPLES, NOT IMPLEMENTED YET) */
typedef enum {
    iio_oflag_none = 0x0,	/* no flags */
    iio_oflag_log = 0x1,	/* log operations on this channel */
    iio_oflag_excl = 0x2	/* open exclusively (NOT IMPLEMENTED) */
} IIO_OFLAG;

    /* return status values */
typedef enum {
    iio_status_ok = 0,		/* all is well */
    iio_status_error = -1,	/* an error */
    iio_status_fatal = -2	/* a serious error */
} IIO_STATUS;

    /* type of module information function */
typedef IIO_STATUS (* IIO_INFOFN)(void);

    /* the user functions */
extern IIO_STATUS
    iio_init(IIO_INFOFN list[], IIO_IFLAG flags),
    iio_open(char *name, IIO_OFLAG flags, IIO *channel),
    iio_close(IIO channel),
    iio_done(void);

    /* family of operate functions */
extern IIO_STATUS
    iio_operate(IIO channel, IIO_OP operation, int data[]),
    iio_operate_real(IIO channel, IIO_OP operation, double data[]),
    iio_operate_addr(IIO channel, IIO_OP operation, void *addr[]);

    /* family of operate functions for inwards simple channels */
extern IIO_STATUS
    iio_operate_in(IIO channel, IIO_OP operation, int data),
    iio_operate_inreal(IIO channel, IIO_OP operation, double data),
    iio_operate_inaddr(IIO channel, IIO_OP operation, void *addr);

    /* error message access function */
extern char
    *iio_emessage_get();

    /* the standard module driver list */
extern IIO_INFOFN iio_standard[];

#ifdef __cplusplus
}
#endif

#endif
