/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: iio.c 2774 2008-03-16 23:56:42Z roy029 $
 *
 * iio.c -- interactive iio testing utility
 * Robin Kirkham, July 1996
 */
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <stdlib.h>

#include "internal.h"

    /* forward declarations */
int iio__help(int argc, char *argv[]);
int iio__tty(FILE *in, int argc, char *argv[]);


char *iio__prompt = "iio> ";
char *iio__prog = "iio";
IIO_BOOL iio__persevere = iio_bool_true;


int iio__operate(int argc, char *argv[]) {
    IIO chan;
    IIO_OP op;
    int fp = 0, count, stat = iio_status_ok;

    /* in/out arrays */
    double ddata[100];
    int idata[100];
    void *adata[100];

    /* open the channel */
    if ((stat = iio_open(argv[1], iio_oflag_log, &chan)))
	return stat;

    /* if no more arguments, print channel details */
    if (! argv[2]) {
	iio_open_show(chan);
	iio_close(chan);
	return iio_status_ok;
    }

    /* otherwise, read the operation code */
    if ((stat = iio_opinfo_lookup(argv[2], &op))) {
	iio_close(chan);
	return stat;
    }

    /* decode the in arguments, if any */
    if (iio_opinfo[op].arg & iio_oparg_in) {
	int number = chan->number;

	/* bitfield ranges take only one in argument */
	if (chan->opnode->chnode->rchnode)
	    number = 1;

	for (count = 0; count < number; ++count) {
	    long ldata;
	    int ecount = 0;

	    /* this is at least something */
	    if (! argv[3 + count]) 
		return iio_error("Insufficient arguments for channel");

	    /* decode arguments into arrays; ignore single errors */
	    if ((stat = iio_decode_double(argv[3 + count], &ddata[count])))
		++ecount;
	    if ((stat = iio_decode_long(argv[3 + count], &ldata)))
		++ecount;
	    if (ecount == 2)
		return stat;

	    idata[count] = (int)ldata;
	    adata[count] = (void *)ldata;

	    /* if any argument has a . assume real data */
	    if (strchr(argv[3 + count], '.'))
		++fp;
	}
    } else
	/* if no in arguments, assume real anyway */
	++fp;


    /* do the operation */
    if (iio_opinfo[op].arg & iio_oparg_addr)
	stat = iio_operate_addr(chan, op, adata);
    else
	if (fp)
	    stat = iio_operate_real(chan, op, ddata);
	else
	    stat = iio_operate(chan, op, idata);

    iio_close(chan);
    return stat;

}

int iio__alias(int argc, char *argv[]) {
    /*
     * Print the mifo list
     */
    return iio_alias_show();
}

int iio__chnode(int argc, char *argv[]) {
    /*
     * Print the chnode list
     */
    return iio_chnode_show(0);
}

int iio__map(int argc, char *argv[]) {
    /*
     * Print the memory map
     */
    return iio_map_show();
}

int iio__minfo(int argc, char *argv[]) {
    /*
     * Print the mifo list
     */
    return iio_minfo_show();
}

int iio__oinfo(int argc, char *argv[]) {
    /*
     * Print the opifo list
     */
    return iio_opinfo_show();
}

int iio__module(int argc, char *argv[]) {
    /*
     * Print the module list
     */
    return iio_module_show();
}

int iio__source(int argc, char *argv[]) {
    /*
     * Read commands from a file, by opening the file, and
     * recursively calling tty()
     */
    FILE *in;
    
    if (! argv[1] || !*argv[1]) 
	return iio_error("Need to specify filename");

    if (! (in = fopen(argv[1], "r")))
	return iio_error(NULL);

    return iio__tty(in, argc, argv);
}

int iio__shell(int argc, char *argv[]) {
    /*
     * Try to shell the given command as a UNIX command
     */
    static char buffer[BUFSIZ];
    int argk;

    /* we actually want the whole line not tokenised */
    *buffer = '\0';
    for (argk = 1; argk < argc; ++argk) {
	strcat(buffer, argv[argk]);
	strcat(buffer, " ");
    }
    return system(buffer);
}

int iio__test(int argc, char *argv[]) {
    /*
     * Just print out arguments out
     */
    int argk;

    printf("Arguments: argc = %d\n", argc);
    for (argk = 0; argk < argc + 1; ++argk)
	printf(
	    "    argv[%02d] = %s%s%s\n", 
	    argk,
	    argv[argk] ? "\"" : "",
	    argv[argk] ? argv[argk] : "<null>",
	    argv[argk] ? "\"" : ""
	);
    return iio_status_ok;
}

int iio__comment(int argc, char *argv[]) {
    /*
     * Comments do nothing
     */
    return iio_status_ok;
}

int iio__quit(int argc, char *argv[]) {
    /*
     * Quit program
     */
    iio__persevere = 0;
    return iio_status_ok;
}


    /* array of functions and names */
static struct {
    char *command;
    int (* function)(int argc, char *argv[]);
    char *help;
} iio__info[] = {
    { "alias",    iio__alias,    "print alias list" },
    { "chnode",   iio__chnode,   "print channel node list" },
    { "map",      iio__map,      "print virtual/physical map" },
    { "minfo",    iio__minfo,    "print module info list" },
    { "module",   iio__module,   "print installed module list" },
    { "oinfo",    iio__oinfo,    "print operation codes" },
    { "operate",  iio__operate,  "operate on a channel" },
    { "source",   iio__source,   "read in run iio commands from a file" },
    { "shell",    iio__shell,    "run UNIX shell command" },
    { "help",     iio__help,     "print this message" },
    { "#",        iio__comment,  "lines beginning with # are ignored" },
#if 0
    { "test",     iio__test,     "test command line" },
#endif
    { "quit",     iio__quit,     "quit program" },
    { NULL, NULL, NULL }
};

int iio__help(int argc, char *argv[]) {
    /*
     * Print the list of functions and help lines
     */
    int index;

    printf("Commands:\n");
    for (index = 0; iio__info[index].command; ++index)
	printf(
	    "    %-10s %s\n",
	    iio__info[index].command,
	    iio__info[index].help
	);
    printf("Commands can be abbreviated\n");
    return iio_status_ok;
}

int iio__tty(FILE *in, int margc, char *margv[]) {
	/*
	 * Tty interface. A mini-shell that accepts a range of commands
	 * and calls a function, passing the arguments in the same way as
	 * to main(). tty() expects to be called by main() with main()'s 
	 * arguments as well
	 */
	int index, nmatch, argc;
	int (* function)(int argc, char *argv[]) = NULL;
	char buffer[BUFSIZ], *line = NULL;
	char *argv[40];

	/* print a banner of sorts */
	if (in == stdin)
		printf(
				"This is %s IIO process\nLibrary timestamp %.0f\n",
				iio_state->init ? "an initial" : "a subsequent",
				iio_timestamp
			  );

	while (iio__persevere) {

		/* print prompt and get command line */
#ifdef HAVE_LIBREADLINE
		extern char *readline(char *prompt);
		extern int add_history(char *line);

		if (line)
			free(line);
		if (! (line = readline(iio__prompt))) 
			break;
		if (line && *line)
			add_history(line);
#else
		if (in == stdin)
			printf(iio__prompt);
		line = buffer;
		if (fgets(line, BUFSIZ-1, in) != line)
			break;
		/* echo if we are reading a file */
		if (in != stdin)
			fputs(line, stdout);
#endif

		/* find first text item, if any */
		argv[0] = strtok(line, " \t\r\n");
		if (!argv[0] || !*argv[0])
			continue;


		/* tokenize the rest of the line */
		for (argc = 1; ; ++argc)
			if (! (argv[argc] = strtok(NULL, " \t\r\n")))
				break;

		/* search the list for a unique match */
		for (nmatch = index = 0; iio__info[index].command; ++index)
			if (strncmp(
						argv[0],
						iio__info[index].command,
						strlen(argv[0])
					   ) == 0) {
				++nmatch;
				argv[0] = iio__info[index].command;
				function = iio__info[index].function;
			}

		/* if more that one possible match, complain */
		if (nmatch > 1) {
			printf("Ambiguous command `%s': Type `help' for help\n", argv[0]);
			continue;
		}

		/* no matches at all */
		if (nmatch == 0) {
			printf("No command `%s': Type `help' for help\n", argv[0]);
			continue;
		}

		/* call the function, passing the full name and arguments */
		switch (function(argc, argv)) {
			case iio_status_ok:
				continue;

			case iio_status_error:
				fprintf(stderr,
						"%s: %s: %s\n",
						argv[0],
						argv[1],
						iio_emessage_get()
					   );
				continue;

			case iio_status_fatal:
				fprintf(stderr, "%s: FATAL ERROR: EXITING\n", iio__prog);
				return 1;
		}
	}
	return iio_status_ok;
}

void iio__interrupt(int sig) {
    /*
     * SIGINT etc signal handler. Cleans up and exits
     */
    fprintf(stderr, "\n%s: signal %d: exiting\n", iio__prog, sig);
    iio_done();
    iio__persevere = 0;
#ifndef VXWORKS
    exit(10);
#endif
}


int main(int argc, char *argv[]) {

    int stat;

    /* where is basename()? */
    if ((iio__prog = strrchr(argv[0], '/'))) 
	++iio__prog;
    else
	iio__prog = argv[0];

    /* start IIO */
    if (iio_init(iio_standard, iio_iflag_none)) {
	fprintf(
	    stderr,
	    "%s: iio_init() failed: %s\n",
	    iio__prog,
	    iio_emessage_get()
	);
	/* iio_done(); */
	return 1;
    }

    /* catch termination signals to exit cleanly */
    signal(SIGHUP, iio__interrupt);
    signal(SIGINT, iio__interrupt);
    signal(SIGQUIT, iio__interrupt);

#if 0
    /* a message */
    printf(
	"\n"
	"WARNING! The syntax of the `operate' command has changed!\n"
	"The operation data type is determined by the argument form, e.g.:\n"
	"    op <chan> <code> 789        does iio_operate(... 789)\n"
	"    op <chan> <code> 0x789      does iio_operate(... 0x789)\n"
	"    op <chan> <code> 789.       does iio_operate_real(... 789.0)\n"
	"\n"
    );
#endif

    /* go to interactive mode */
    stat = iio__tty(stdin, argc, argv);

    iio_done();
    return stat;
}
