/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: data.c 365 2003-05-12 02:31:19Z sik057 $
 *
 * data.c -- user data get/set functions for module drivers
 * Robin Kirkham, January 1997
 */
#include "internal.h"

#define LIMIT(value, upper, lower) \
    if (upper > lower) { \
	if (value > upper) \
	    value = upper; \
	else \
	    if (value < lower) \
		value = lower; \
    }


HIDDEN IIO_STATUS iio_data_log(
    IIO_OPNODE *opnode, unsigned seqno, char *format, ...
) {
    /* 
     * Logging funcion called by the various iio_data...() functions. 
     * This logs a standard leader identifying the channel, operation,
     * and user data type, followed by the results of formatting the format
     * string with its appended arguments
     */
    char buffer[200];
    va_list ap;
    va_start(ap, format);

    /* format the variable part */
    iio_vslog(buffer, format, ap);
    
    /* log the whole lot */
    iio_log("    %s.%d:%s%d.%d: %s\n",
	opnode->chnode->module->minfo->ident,
	opnode->chnode->module->seq,
	iio_chtype_string[opnode->chnode->type],
	opnode->chnode->width,
	seqno,
	buffer
    );
    return iio_status_ok;
}


int iio_data_get(IIO_OPNODE *opnode, unsigned seqno) {
    /*
     * Index the user data as an array of whatever type it is, get the
     * value indicated by seqno, and convert it (if necessary) to an int
     * and return it. The conversion is subject to the scale/offset values
     * and limits for that channel
     */
    int value;

    switch (opnode->udata) {
    case iio_udata_real:
	value = iio_round(
	    (((double *)opnode->base)[seqno + opnode->index] -
		opnode->chnode->chinfo[seqno].offset
	    )/opnode->chnode->chinfo[seqno].scale
	);
	LIMIT(
	    value, 
	    opnode->chnode->chinfo[seqno].upper,
	    opnode->chnode->chinfo[seqno].lower
	);
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"ri[%d]: %08x <- %g %s",
		seqno + opnode->index,
		(unsigned)value, 
		opnode->chnode->chinfo[seqno].scale * value +
		    opnode->chnode->chinfo[seqno].offset,
		opnode->chnode->chinfo[seqno].unit
	    );
	return value;

    case iio_udata_int:
	value = ((int *)opnode->base)[seqno + opnode->index];
	LIMIT(
	    value, 
	    opnode->chnode->chinfo[seqno].upper,
	    opnode->chnode->chinfo[seqno].lower
	);
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"ii[%d]: %08x <- %d",
		seqno + opnode->index,
		(unsigned)value, 
		((int *)opnode->base)[seqno + opnode->index]
	    );
	return value;

    case iio_udata_bit:
	if (opnode->log)
	    iio_log("        %s.%d:%s%d.%d: %08x (&%08x) <-\n",
		opnode->chnode->rchnode->module->minfo->ident,
		opnode->chnode->rchnode->module->seq,
		iio_chtype_string[opnode->chnode->rchnode->type],
		opnode->chnode->rchnode->width,
		opnode->chnode->rseqno,
		(unsigned)*((int *)opnode->base),
		((1 << opnode->number) - 1) << (opnode->first + opnode->index)
	    );
	return *((int *)opnode->base);

    default:
    case iio_udata_addr:
	return iio_fatal("iio_data_get: access of inappropriate type");
    }
}

double iio_data_get_real(IIO_OPNODE *opnode, unsigned seqno) {
    /*
     * Index the user data as an array of whatever type it is, get the
     * value indicated by seqno, and convert it (if necessary) to an double
     * and return it. The conversion is subject to the scale/offset values
     * and limits for that channel
     */
    double value;

    switch (opnode->udata) {
    case iio_udata_real:
	value = (
	    (((double *)opnode->base)[seqno + opnode->index] -
		opnode->chnode->chinfo[seqno].offset
	    )/opnode->chnode->chinfo[seqno].scale
	);
	LIMIT(
	    value, 
	    opnode->chnode->chinfo[seqno].upper,
	    opnode->chnode->chinfo[seqno].lower
	);
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"rr[%d]: %g <- %g %s",
		seqno + opnode->index,
		value, 
		opnode->chnode->chinfo[seqno].scale * value +
		    opnode->chnode->chinfo[seqno].offset,
		opnode->chnode->chinfo[seqno].unit
	    );
	return value;

    case iio_udata_int:
	value = (double)((int *)opnode->base)[seqno + opnode->index];
	LIMIT(
	    value, 
	    opnode->chnode->chinfo[seqno].upper,
	    opnode->chnode->chinfo[seqno].lower
	);
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"ir[%d]: %g <- %d",
		seqno + opnode->index,
		value, 
		((int *)opnode->base)[seqno + opnode->index]
	    );
	return value;

    default:
    case iio_udata_bit:
    case iio_udata_addr:
	return iio_fatal("iio_data_get_real: access of address type");
    }
}

void *iio_data_get_addr(IIO_OPNODE *opnode, unsigned seqno) {
    /*
     * Index the user data as an array and return the element indicated 
     * by seqno. No conversions are performed. This function can only be
     * used to get an address type value from a \n address type operation
     * function call (i.e. iio_operate_addr())
     */
#if 0
    if (opnode->udata != iio_udata_addr)
	return iio_fatal("iio_data_get_addr: access of non-address type");
#endif

    if (opnode->log)
	iio_data_log(opnode, seqno,
	    "aa[%d]: %08x <- %08x",
	    seqno + opnode->index,
	    (unsigned)((void **)opnode->base)[seqno + opnode->index],
	    (unsigned)((void **)opnode->base)[seqno + opnode->index]
	);
    return ((void **)opnode->base)[seqno + opnode->index];
}


IIO_STATUS iio_data_set(IIO_OPNODE *opnode, unsigned seqno, int value) {
    /*
     * Index the user data as an array of whatever type it is, and set the
     * element indicated by seqno to value. No conversions are performed,
     * unless the user data type is real, in which the scale/offset factors
     * apply (but not the limits)
     */
    switch (opnode->udata) {
    case iio_udata_real:
	((double *)opnode->base)[seqno + opnode->index] =
	    opnode->chnode->chinfo[seqno].scale * value +
	    opnode->chnode->chinfo[seqno].offset;
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"ir[%d]: %08x -> %g %s",
		seqno + opnode->index,
		(unsigned)value, 
		((double *)opnode->base)[seqno + opnode->index],
		opnode->chnode->chinfo[seqno].unit
	    );
	return iio_status_ok;

    case iio_udata_int:
	((int *)opnode->base)[seqno + opnode->index] = value;
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"ii[%d]: %08x -> %d",
		seqno + opnode->index,
		(unsigned)value, 
		((int *)opnode->base)[seqno + opnode->index]
	    );
	return iio_status_ok;

    case iio_udata_bit:
	if (opnode->log)
	    iio_log("        %s.%d:%s%d.%d: %08x (&%08x) ->\n",
		opnode->chnode->rchnode->module->minfo->ident,
		opnode->chnode->rchnode->module->seq,
		iio_chtype_string[opnode->chnode->rchnode->type],
		opnode->chnode->rchnode->width,
		opnode->chnode->rseqno,
		(unsigned)value,
		((1 << opnode->number) - 1) << (opnode->first + opnode->index)
	    );
	*(int *)opnode->base = value;
	return iio_status_ok;

    default:
    case iio_udata_addr:
	return iio_fatal("iio_data_get_real: access of address type");
    }
}

IIO_STATUS iio_data_set_real(IIO_OPNODE *opnode, unsigned seqno, double value) {
    /*
     * Index the user data as an array of whatever type it is, and set the
     * element indicated by seqno to value. The scale/offset factors always
     * apply (but not the limits)
     */
    switch (opnode->udata) {
    case iio_udata_real:
	((double *)opnode->base)[seqno + opnode->index] =
	    opnode->chnode->chinfo[seqno].scale * value +
	    opnode->chnode->chinfo[seqno].offset;
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"rr[%d]: %g -> %g %s",
		seqno + opnode->index,
		value, 
		((double *)opnode->base)[seqno + opnode->index],
		opnode->chnode->chinfo[seqno].unit
	    );
	return iio_status_ok;

    case iio_udata_int:
	((int *)opnode->base)[seqno + opnode->index] = iio_round(value);
	if (opnode->log)
	    iio_data_log(opnode, seqno,
		"ri[%d]: %g -> %d",
		seqno + opnode->index,
		value, 
		((int *)opnode->base)[seqno + opnode->index]
	    );
	return iio_status_ok;

    default:
    case iio_udata_addr:
	return iio_fatal("iio_data_get_real: access of address type");
    }
}

IIO_STATUS iio_data_set_addr(IIO_OPNODE *opnode, unsigned seqno, void *value) {
    /*
     * Index the user data as an array and set the element indicated 
     * by seqno to value. No conversions are performed. This function can
     * only be used to set an address type value from an address type operation
     * function call (i.e. iio_operate_addr())
     */
    if (opnode->udata != iio_udata_addr)
	return iio_fatal("iio_data_get_addr: access of non-address type");
    
    ((void **)opnode->base)[seqno + opnode->index] = value;
    if (opnode->log)
	iio_data_log(opnode, seqno,
	    "aa[%d]: %08x -> %08x",
	    seqno + opnode->index,
	    (unsigned)value, 
	    (unsigned)((void **)opnode->base)[seqno + opnode->index]
	);
    return iio_status_ok;
}
