/*
 * This file is part of IIO, the Industrial IO Library
 * CSIRO Division of Manufacturing Technology
 * $Id: chtype.c 2222 2007-12-04 11:46:42Z roy029 $
 *
 * chtype.c -- string arrays for channel type
 * Robin Kirkham, June 1996
 */
#include "internal.h"

    /* matches IIO_CHTYPE */
char *iio_chtype_string[] = {
    "di",
    "do",
    "dio",
    "oco",
    "ocio",

    "bi",
    "bo",
    "bio",
    "boco",
    "bocio",

    "adc",
    "dac",
    "ctr",
    "reg",
    "enc",
    "rdio",

    "isa",
    "ip",
    "null",
    "vme",
    "sc",
    "adam",
    NULL	/* NULL must be last */
};


IIO_STATUS iio_chtype_find(char *string, IIO_CHTYPE *type, unsigned *width) {
    /*
     * Decode a channel type string. The channel type consists of one of the 
     * legal channel types (in iio_chtype_string[]) optionally followed by
     * the width of the channel, without intervening characters. The channel
     * type is returned into type, and the width into width. If there was no
     * width given, the width is zero (which usually means any width will do).
     * Returns error status if the name does not match or the width is too big.
     */
    char *pwidth;
    long lwidth = 0;
    int nlength = 0, count;

    /* check some arguments */
    if (!string || !*string)
	return iio_fatal("NULL or empty channel type string");
    if (!type)
	return iio_fatal("NULL channel type return pointer");
    if (!width)
	return iio_fatal("NULL width return pointer");


    /* first try and find the width digits */
    if ((pwidth = iio_string_pbrk(string, "123456789"))) {

	/* decode the width number */
	iio_eret( iio_decode_long(pwidth, &lwidth) );

	/* zero or large width not permitted */
	if (lwidth < 1 || lwidth > channelwidth(int))
	    return iio_error("Channel width out of range");

	/* get the length of the name part of the string */
	nlength = (int)(pwidth - string);

    } else {

	/* if no width given, the whole string is the type */
	nlength = iio_string_len(string);
	lwidth = 0;
    }


    /*
     * Now look up the name part in the list of channel names. Since we
     * can't change the string, we can't terminate it, so we can't use
     * strcmp(), so we do stuff with the string length and srtncmp()
     */
    for (count = 0; iio_chtype_string[count]; ++count)
	if (
	    iio_string_len(iio_chtype_string[count]) == nlength &&
	    (iio_string_ncmp(iio_chtype_string[count], string, nlength) == 0)
	) {
	    /* return the results */
	    *width = (unsigned)lwidth;
	    *type = (IIO_CHTYPE)count;
	    return iio_status_ok;
	}

    /* not found */
    return iio_error("No such channel type");
}
